unit Helpers;

// This project is documented using the Doxygen documentation system.
// http://www.doxygen.org
// http://pas2dox.sourceforge.net

{* @file Helpers.pas
 * @brief Various helper functions.
}

interface

uses
  Windows, DateUtils, SysUtils, Classes, Contnrs, Math,
  Config, dEXIF;

    // Helper functions for conversions and write-protection
    function NumStri(var arg: String): Integer;
    function Stri(number: Integer): String;
    function ReplaceString(arg, oldString, newString: String): String;
    function DTToFT(const ADateTime: TDateTime): TFileTime;
    function FTToDT(const AFileTime: TFileTime): TDateTime;
    function Unprotect(filename: String; var Attrs: Integer): Boolean;
    procedure Protect(filename: String; Attrs: Integer);

    // created/modified/accessed timestamp from file system
    function GetFsTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean): Boolean;
    function SetFsTimestamp(filename: String; timestamp: MyTimestamp; isDirectory, byConfig: Boolean): Boolean;

    procedure UpdateDT(var oldDateTime: TDateTime; newDateTime: TDateTime);
    //procedure UpdateMT(var oldMyTimestamp: MyTimestamp; newMyTimestamp: MyTimestamp);

    // get/set a complete timestamp with all values which are selected in options
    function GetMyTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;
    function SetMyTimestamp(filename: String; timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;

    // called by Unit1 on single files/dirs
    function getFileDateTime(filename: String; var DateTime: TDateTime): Boolean;
    function getDirDateTime(filename: String; var DateTime: TDateTime): Boolean;
    function setFileDateTime(filename: String; DateTime: TDateTime): Boolean;
    function setDirDateTime(filename: String; DateTime: TDateTime): Boolean;

    function IncDateTime(DateTime: TDateTime; inc: MyIncTime): TDateTime;

    function SetList(list: TStrings; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
    function SetRecursiveBF(directory: String; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
    function SetRecursive(directory: String; inc: MyIncTime; var DateTime: TDateTime): MyErrors;

    procedure InitDatesList();
    procedure DestroyDatesList();

    function RememberList(list: TStrings): MyErrors;
    function RememberRecursive(directory: String): MyErrors;
    function SaveRemembered(): MyErrors;
    procedure ForgetRemembered();

    procedure InitTimestamp(var timestamp: MyTimestamp);
    procedure InitMyErrors(var errs: MyErrors);
    procedure AddMyErrors(var target: MyErrors; source: MyErrors);
    function MaxMyErrors(errs1: MyErrors; errs2: MyErrors): MyErrors;
    procedure SetMyTimestampToolOpts(filename: String; inc: MyIncTime; var DateTime: TDateTime; isDirectory: Boolean; var errs: MyErrors);

const
  fdDefault: Integer = 0;


implementation

{*
 * Validate String number (remove non-number chars).
 * @param arg A string containing a number.
 * @return The number (0 if String was empty).
 }
function NumStri(var arg: String): Integer;
var
  code, number:Integer;
begin
  if arg='' then begin
    number := 0;
  end else begin
    VAL(arg, number, code);
    Str(number, arg);
  end;
  Result := number;
end;


{*
 * Convert Integer to String.
 * @param number A number.
 * @return The number as string.
 }
function Stri(number: Integer): String;
var arg: string;
begin
  Str(number, arg);
  Stri := arg;
end;


{*
 * Replace all occurrences of a substring.
 * @param arg The string.
 * @param oldString The substring to be replaced.
 * @param newString The replacement substring.
 }
function ReplaceString(arg, oldString, newString: String): String;
var
  tmp: String;
begin
  tmp := '';

  while Pos(oldString, arg) > 0 do begin
    tmp := tmp + copy(arg, 1, Pos(oldString, arg)-1);
    tmp := tmp + newString;
    Delete(arg, 1, Pos(oldString, arg)-1 + Length(oldString));
  end;

  Result := tmp + arg;
end;


{*
 * Convert TDateTime to TFileTime.
 * Thanks to
 * https://forums.codegear.com/thread.jspa?threadID=11125&tstart=0
 * @param ADateTime The time as TDateTime.
 * @return The time as TFileTime.
 }
function DTToFT(const ADateTime: TDateTime): TFileTime;
var
ft: TFileTime;
st: TSystemTime;
begin
DateTimeToSystemTime(ADateTime, st);
SystemTimeToFileTime(st, ft);
LocalFileTimeToFileTime(ft, ft);
Result := ft;
end;


{*
 * Convert TFileTime to TDateTime.
 * @param AFileTime The time as TFileTime.
 * @return The time as TDateTime.
 }
function FTToDT(const AFileTime: TFileTime): TDateTime;
var
ft: TFileTime;
st: TSystemTime;
dt: TDateTime;
begin
ft := AFileTime;
FileTimeToLocalFileTime(ft, ft);
FileTimeToSystemTime(ft, st);
dt := SystemTimeToDateTime(st);
Result := dt;
end;


{*
 * Remove write protection.
 * @param filename The file path.
 * @param Attrs The original file attributes.
 * @return true if unprotected, false otherwise.
 }
function Unprotect(filename: String; var Attrs: Integer): Boolean;
var err: Integer;
begin
  Result := true;
  Attrs := FileGetAttr(filename);
  if not (Attrs and faReadOnly = 0) then begin
    if Config.OptWriteProtected then begin
      err := FileSetAttr(filename, Attrs - faReadOnly);
      if not err=0 then Result := false;
    end else
      Result := false;
  end;
end;


{*
 * Restore write protection.
 * @param filename The file path.
 * @param Attrs The original file attributes.
 }
procedure Protect(filename: String; Attrs: Integer);
begin
  if (Attrs and faReadOnly <> 0) and (Config.OptWriteProtected) then begin
      FileSetAttr(filename, Attrs);
  end;
end;


{*
 * Get a file's / directory's timestamp from file system.
 * (Windows 95/98/ME: No directories supported!)
 }
function GetFsTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean): Boolean;
var F: THandle;
  AgeCreated, AgeAccess, AgeModified: TFileTime;
begin
  Result := False;

  // get file handle
  if isDirectory then begin
    F := CreateFileA(PAnsiChar(FileName), GENERIC_READ,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, 0);
  end else begin //isFile
    F := CreateFileA(PAnsiChar(filename), GENERIC_READ,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, 0, 0);
  end;
  if F = INVALID_HANDLE_VALUE then Exit;

  // get file time
  Result := GetFileTime(F, @AgeCreated, @AgeAccess, @AgeModified);

  timestamp.created := FTToDT(AgeCreated);
  timestamp.modified := FTToDT(AgeModified);
  timestamp.access := FTToDT(AgeAccess);

  CloseHandle(F);
end;


{*
 * Set a file's / directory's timestamp in the file system.
 * (Windows 95/98/ME: No directories supported!)
 * If param byConfig is set, the particular data will be set depending on:
 * - Config.OptTarget_Created
 * - Config.OptTarget_Modified
 * - Config.OptTarget_Access
 *
 * @param filename The path.
 * @param timestamp The timestamp.
 * @param isDirectory true if directory, false if file.
 * @param byConfig true to save only activated datetimes,
 *   false to save complete timestamp.
 * @return true if successful, false otherwise.
 }
function SetFsTimestamp(filename: String; timestamp: MyTimestamp; isDirectory, byConfig: Boolean): Boolean;
var F: THandle;
  AgeCreated, AgeAccess, AgeModified: TFileTime;
begin
  Result := false;

  // get file handle
  if isDirectory then begin
    F := CreateFileA(PAnsiChar(FileName), GENERIC_WRITE,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, 0);
  end else begin //isFile
    F := CreateFileA(PAnsiChar(FileName), GENERIC_WRITE,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, 0, 0);
  end;
  if F = INVALID_HANDLE_VALUE then Exit;

  AgeCreated := DTToFT(timestamp.created);
  AgeAccess := DTToFT(timestamp.access);
  AgeModified := DTToFT(timestamp.modified);

  // set file time
  if byConfig then begin
    Result := true;
    if Config.OptTarget_Created then {CreateTime} Result := Result and SetFileTime(F, @AgeCreated, nil, nil);
    if Config.OptTarget_Access then {LastAccessTime} Result := Result and SetFileTime(F, nil, @AgeAccess, nil);
    if Config.OptTarget_Modified then {LastWriteTime}  Result := Result and SetFileTime(F, nil, nil, @AgeModified);
  end else begin
    Result := SetFileTime(F, @AgeCreated, @AgeAccess, @AgeModified);
  end;

  CloseHandle(F);
end;


{*
 * Updates date and/or time depending on the user's choice.
 * (Config.OptDate and Config.OptTime are set by Checkbox_Date and Checkbox_Time)
 * @param oldDateTime The datetime to be updated.
 * @param newDateTime The new datetime to be set.
 }
procedure UpdateDT(var oldDateTime: TDateTime; newDateTime: TDateTime);
var dt: TDateTime;
begin
  // A zero value shall not be written (newDateTime may be zero if EXIF data is missing).
  if(newDateTime=0) then exit;
  // Frac returns time only, Trunc returns date only
  if Config.OptTime then dt := Frac(newDateTime) else dt := Frac(oldDateTime);
  if Config.OptDate then dt := dt + Trunc(newDateTime) else dt := dt + Trunc(oldDateTime);
  oldDateTime := dt;
end;


{*
 * Get the timestamp which is selected in Options => Source.
 * (Maybe multiple timestamps if "each selected target date" is chosen.)
 * The filesystem timestamp (created, modified, access) will always be retrieved.
 * @param filename The file or directory path.
 * @param timestamp Will contain the timestamps thereafter.
 * @param isDirectory true if directory, false if file
 * @param errs Returned errors.
 * @return true if at least created/modified/access timestamp were read successfully, false otherwise.
 }
function GetMyTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;
var
  ImgData: TImgData;
  Attrs: Integer;
  exif: Boolean;
begin
  Result := false;
  //errs.files := errs.files + 1;
  //if not FileExists(filename) then exit;

  InitTimestamp(timestamp);

  // FsTimestamp will always be retrieved!
  // (This is needed to preserve the FsTimestamp when only the file contents
  // should be changed, e.g. the EXIF timestamp.)
  Result := GetFsTimestamp(filename, timestamp, isDirectory);
  if not Result then begin
    // Access error (maybe file in use?)
    if isDirectory then begin
      errs.dirAccess := errs.dirAccess + 1;
    end else begin
      errs.fileAccess := errs.fileAccess + 1;
    end;
    exit;
  end;

  if not isDirectory then begin // isFile

    // In order to allow only date xor time being changed,
    // we must read EXIF data even if it should only be handled as target:
    if (Config.OptSource_EXIF_DateTime or Config.OptSource_EXIF_DateTimeOriginal or Config.OptSource_EXIF_DateTimeDigitized)
     or (Config.OptTarget_EXIF_DateTime or Config.OptTarget_EXIF_DateTimeOriginal or Config.OptTarget_EXIF_DateTimeDigitized)
     then begin
      exif := false;
      try
        ImgData := TimgData.Create();
        ImgData.BuildList := dEXIF.GenAll;

        if ImgData.ProcessFile(filename) then begin
          if ImgData.HasEXIF() then begin
            timestamp.exifDT := ImgData.ExifObj.GetImgDT();
            timestamp.exifDTO := ImgData.ExifObj.GetImgDTO();
            timestamp.exifDTD := ImgData.ExifObj.GetImgDTD();
            exif := true;
          end;
        end;
      finally
        if Unprotect(filename, Attrs) then begin
          // Touching the file contents seems to change the last access date.
          // Thus, we need to reset it:
          SetFsTimestamp(filename, timestamp, isDirectory, false);
          Protect(filename, Attrs);
          // However, we cannot prohibit the Windows Explorer from
          // accessing/reloading a file whenever it was changed.
        end else begin
          errs.wprotected := errs.wprotected + 1;
        end;
        if not exif then errs.exifRead := errs.exifRead + 1;
      end;
    end;

  end;

end;


{*
 * Set the timestamp/s which is/are selected in Options => Target.
 * @param filename The file or directory path.
 * @param myTimestamp The new timestamp/s.
 * @param isDirectory true if directory, false if file
 * @param errs Returned errors.
 * @return true if successful, false otherwise.
 }
function SetMyTimestamp(filename: String; timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;
var
  ImgData: TImgData;
  fileTimestamp: MyTimestamp;
begin
  Result := false;

  //errs.files := errs.files + 1;
  //if not FileExists(filename) then exit;

  // Remember: FsTimestamp MUST be read, as file may be edited for EXIF data and
  // may loose its original timestamp otherwise!

  // fetch old timestamp/s for adjustment
  Result := GetMyTimestamp(filename, fileTimestamp, isDirectory, errs);
  if not Result then exit;

  // these three values should be set at once
  if Config.OptTarget_Created then UpdateDT(fileTimestamp.created, timestamp.created);
  if Config.OptTarget_Modified then UpdateDT(fileTimestamp.modified, timestamp.modified);
  if Config.OptTarget_Access then UpdateDT(fileTimestamp.access, timestamp.access);

  // Please note: UpdateDT is used here to modify the current target datetime
  // in order to preserve the target's original time or date
  // (if the user wants to change only the time OR only the date).

  // Special option: Directories: Additionally set created date
  // (if I recall correctly, Windows Explorer used to show created date for dirs)
  if Config.OptSetDirCreateTime and Config.OptTarget_Modified
   and (not Config.OptTarget_Created) then begin
    UpdateDT(fileTimestamp.created, timestamp.modified);
  end;

  // EXIF date may only be available for files, not for directories ;)
  if not isDirectory then begin // isFile
    if (Config.OptTarget_EXIF_DateTime or Config.OptTarget_EXIF_DateTimeOriginal or Config.OptTarget_EXIF_DateTimeDigitized) then begin
      Result := false;
      try
        ImgData := TimgData.Create();
        ImgData.BuildList := dEXIF.GenAll;

        if ImgData.ProcessFile(filename) then begin
          if ImgData.HasEXIF() then begin
            if (Config.OptTarget_EXIF_DateTime) then begin
              UpdateDT(fileTimestamp.exifDT, timestamp.exifDT);
              ImgData.ExifObj.OverwriteDT(fileTimestamp.exifDT);
            end;
            if (Config.OptTarget_EXIF_DateTimeOriginal) then begin
              UpdateDT(fileTimestamp.exifDTO, timestamp.exifDTO);
              ImgData.ExifObj.OverwriteDTO(fileTimestamp.exifDTO);
            end;
            if (Config.OptTarget_EXIF_DateTimeDigitized) then begin
              UpdateDT(fileTimestamp.exifDTD, timestamp.exifDTD);
              ImgData.ExifObj.OverwriteDTD(fileTimestamp.exifDTD);
            end;
            ImgData.WriteEXIFJpeg(filename);
            Result := true;
          end;
        end;
      finally
        if not Result then errs.exifWrite := errs.exifWrite+1;
      end;

    end;

  end;

  // This MUST be the last operation, as any editing may destroy
  // the FsTimestamp!
  Result := SetFsTimestamp(filename, fileTimestamp, isDirectory, false) and Result;

end;


{*
 * called by OpenFileDateNow (single file only):
 * Get date and time for file.
 * @param FileName The file path.
 * @param DateTime The file's timestamp.
 * @return true if successful, false otherwise.
 }
function getFileDateTime(FileName: String; var DateTime: TDateTime): Boolean;
var
  timestamp: MyTimestamp;
  errs: MyErrors;
begin
  InitMyErrors(errs);
  Result := GetMyTimestamp(filename, timestamp, false, errs);
  // Exif errors must be detected separately:
  if ((errs.exifRead > 0) and
   (Config.OptSource_EXIF_DateTime or
    Config.OptSource_EXIF_DateTimeOriginal or
    Config.OptSource_EXIF_DateTimeDigitized) ) then
   Result := false;
  if Result then begin
    if Config.OptSource_Created then DateTime := timestamp.created
    else if Config.OptSource_Modified then DateTime := timestamp.modified
    else if Config.OptSource_Access then DateTime := timestamp.access
    else if Config.OptSource_EXIF_DateTime then DateTime := timestamp.exifDT
    else if Config.OptSource_EXIF_DateTimeOriginal then DateTime := timestamp.exifDTO
    else if Config.OptSource_EXIF_DateTimeDigitized then DateTime := timestamp.exifDTD
    else Result := false;
  end;
end;


{*
 * called by OpenDirDateNow (single file only):
 * Get date and time for directory.
 * @param filename The directory path.
 * @param DateTime The file's timestamp.
 * @return true if successful, false otherwise.
 }
function getDirDateTime(filename: String; var DateTime: TDateTime): Boolean;
var
  timestamp: MyTimestamp;
  errs: MyErrors;
begin
  InitMyErrors(errs);
  Result := GetMyTimestamp(filename, timestamp, true, errs);
  if Result then begin
    if Config.OptSource_Created then DateTime := timestamp.created
    else if Config.OptSource_Modified then DateTime := timestamp.modified
    else if Config.OptSource_Access then DateTime := timestamp.access
    //else if Config.OptSource_EXIF then //ignore
    else Result := false;
  end;
end;


{*
 * called by SaveDateNow (single file only):
 * save datetime to file
 * @param filename The file path.
 * @param DateTime The file's new timestamp.
 * @return true if successful, false otherwise.
 }
function setFileDateTime(filename: String; DateTime: TDateTime): Boolean;
var
  timestamp: MyTimestamp;
  errs: MyErrors;
begin
  InitMyErrors(errs);
  Result := true;
  if Config.OptTarget_Created then timestamp.created := DateTime;
  if Config.OptTarget_Modified then timestamp.modified := DateTime;
  if Config.OptTarget_Access then timestamp.access := DateTime;
  if Config.OptTarget_EXIF_DateTime then timestamp.exifDT := DateTime;
  if Config.OptTarget_EXIF_DateTimeOriginal then timestamp.exifDTO := DateTime;
  if Config.OptTarget_EXIF_DateTimeDigitized then timestamp.exifDTD := DateTime;
  Result := SetMyTimestamp(filename, timestamp, false, errs);
end;


{*
 * save datetime to directory
 * @param filename The directory path.
 * @param DateTime The directory's new timestamp.
 * @return true if successful, false otherwise.
 }
function setDirDateTime(filename: String; DateTime: TDateTime): Boolean;
var
  timestamp: MyTimestamp;
  errs: MyErrors;
begin
  InitMyErrors(errs);
  Result := true;
  if Config.OptTarget_Created then timestamp.created := DateTime
  else if Config.OptTarget_Modified then timestamp.modified := DateTime
  else if Config.OptTarget_Access then timestamp.access := DateTime
  else Result := false;
  if not Result then exit;
  Result := SetMyTimestamp(filename, timestamp, true, errs);
end;


{*
 * Increase the values of a TDateTime object.
 * Each part of the timestamp (second, minute, ...) will be increased
 * individually.
 * @param DateTime The timestamp to be increased.
 * @param inc The increase time.
 * @return The increased timestamp.
 }
function IncDateTime(DateTime: TDateTime; inc: MyIncTime): TDateTime;
begin
  DateTime := IncSecond(DateTime,inc.s);
  DateTime := IncMinute(DateTime,inc.m);
  DateTime := IncHour(DateTime,inc.h);
  DateTime := IncDay(DateTime,inc.d);
  DateTime := IncYear(DateTime,inc.a);
  Result := DateTime;
end;


{*
 * Set dates for multiple files.
 * @param list A list of files.
 * @param inc The increase time.
 * @param DateTime The new timestamp.
 * @return The error count.
 }
function SetList(list: TStrings; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
var
  timestamp: MyTimestamp;
  count, Attrs{, CurrentDate}:integer;
begin
  InitMyErrors(Result);
  Result.files := list.Count;

  For count := 0 to list.Count-1 do begin
    //process file
    SetMyTimestampToolOpts(list[count], inc, DateTime, false, Result);
  end;
end;


{*
 * Set dates recursively (breadth first).
 * @param directory The directory path.
 * @param inc The increase time.
 * @param DateTime The new timestamp.
 * @return The error count.
 }
function SetRecursiveBF(directory: String; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
var
  filenameptr, filenameptr2: ^String;
  SearchRec: TSearchRec;
  WorkList: TQueue;
  DirList: TStringList;
  FileList: TStringList;
  count: Integer;
  {CurrentDate,} Attrs: Integer;
  timestamp: MyTimestamp;
label
  AddDirList, AddFileList, EndAddLists;
begin
  InitMyErrors(Result);

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  New(filenameptr);
  filenameptr^ := directory;

  WorkList := TQueue.Create;
  WorkList.Push(filenameptr);

  while WorkList.Count>0  do begin
    filenameptr := WorkList.Pop;
    if(filenameptr^[length(filenameptr^)] = '\') then begin

      if Config.OptToolsSetDirTime then begin
        //process directory
        Result.dirs := Result.dirs + 1;
        SetMyTimestampToolOpts(filenameptr^, inc, DateTime, true, Result);
      end;

      //recurse subdirs (breadth first)

      DirList := TStringList.Create;
      FileList := TStringList.Create;

      if FindFirst(filenameptr^ + '*.*', faDirectory+faReadOnly, SearchRec) = 0 then begin
        repeat
          if (SearchRec.Attr and faDirectory = faDirectory)
          and (not (SearchRec.Name[1] = '.')) then begin
            //Current entry is a directory
            DirList.Add(filenameptr^+SearchRec.Name);
          end else begin
            //entry is a file
            if not (SearchRec.Name[1] = '.') then begin
              FileList.Add(filenameptr^+SearchRec.Name);
            end;
          end;
        until (not (FindNext(SearchRec) = 0));
        FindClose(SearchRec);

        // A little "goto" programming for alternate order.
        if Config.OptFilesFirst then goto AddFileList;

        AddDirList:
        DirList.Sort;
        for count := 0 to DirList.Count-1 do begin
          // We are not yet going recursive here... (i.e. breadth first)
          New(filenameptr2);
          filenameptr2^ := DirList.Strings[count];
          if not (filenameptr2^[length(filenameptr2^)] = '\') then
            filenameptr2^ := filenameptr2^ + '\';
          WorkList.Push(filenameptr2);
        end;
        DirList.Free;
        if Config.OptFilesFirst then goto EndAddLists;

        AddFileList:
        FileList.Sort;
        for count := 0 to FileList.Count-1 do begin
          // Also no direct processing here, but adding file to the queue
          New(filenameptr2);
          filenameptr2^ := FileList.Strings[count];
          WorkList.Push(filenameptr2);
        end;
        FileList.Free;
        if Config.OptFilesFirst then goto AddDirList;

        EndAddLists:

      end;


    end else begin
      //process file
      Result.files := Result.files + 1;
      SetMyTimestampToolOpts(filenameptr^, inc, DateTime, false, Result);
    end;
    Dispose(filenameptr);

  end;

end;



{*
 * Set dates recursively (depth first).
 * @param directory The directory path.
 * @param inc The increase time.
 * @param DateTime The new timestamp.
 * @return The error count.
 }
function SetRecursive(directory: String;
  inc: MyIncTime;
  var DateTime: TDateTime): MyErrors;
var
  SearchRec: TSearchRec;
  DirList: TStringList;
  FileList: TStringList;
  count: Integer;
  {CurrentDate,} Attrs: Integer;
  errs: MyErrors;
  timestamp: MyTimestamp;
label
  AddDirList, AddFileList, EndAddLists;
begin
  InitMyErrors(Result);
  InitMyErrors(errs);

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  if Config.OptToolsSetDirTime then begin
    // process directory
    Result.dirs := Result.dirs+1;
    SetMyTimestampToolOpts(directory, inc, DateTime, true, Result);
  end;

  //recurse subdirs (depth first)

  DirList := TStringList.Create;
  FileList := TStringList.Create;

  if FindFirst(directory + '*.*', faDirectory+faReadOnly, SearchRec) = 0 then begin
    repeat
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        DirList.Add(directory+SearchRec.Name);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          FileList.Add(directory+SearchRec.Name);
        end;
      end;
    until (not (FindNext(SearchRec) = 0));
    FindClose(SearchRec);

    // A little "goto" programming for alternate order.
    if Config.OptFilesFirst then goto AddFileList;

    AddDirList:
    DirList.Sort;
    for count := 0 to DirList.Count-1 do begin
      // process directory recursively (i.e. depth first)
      errs := SetRecursive(DirList.Strings[count],inc,DateTime);
      AddMyErrors(Result,errs);
    end;
    DirList.Free;
    if Config.OptFilesFirst then goto EndAddLists;

    AddFileList:
    FileList.Sort;
    for count := 0 to FileList.Count-1 do begin
      // process file
      Result.files := Result.files + 1;
      SetMyTimestampToolOpts(FileList.Strings[count], inc, DateTime, false, Result);
    end;
    FileList.Free;
    if Config.OptFilesFirst then goto AddDirList;

    EndAddLists:

  end;

end;


{*
 * Init list for remembering dates.
 }
procedure InitDatesList();
begin
  if DatesList<>nil then try
    DatesList.Free;
  finally
  end;
  DatesList := TList.Create;
end;


{*
 * Destroy list for remembering dates.
 }
procedure DestroyDatesList();
var
  count: Integer;
begin
  if DatesList<>nil then begin
    try
      for count := 0 to DatesList.Count - 1 do begin
        Dispose(DatesList.Items[count]);
      end;
    finally
      DatesList.Free;
    end;
  end;
end;


{*
 * Store dates in list object.
 * @param list A list of files.
 }
function RememberList(list: TStrings): MyErrors;
var count:integer;
  datefileptr: ^MyDateFile;
  timestamp: MyTimestamp;
  errs: MyErrors;
begin
  InitMyErrors(errs);
  InitDatesList();

  for count := 0 to list.Count-1 do begin
    if GetMyTimestamp(list[count], timestamp, false, errs) then begin
      New(datefileptr);
      datefileptr^.timestamp := timestamp;
      datefileptr^.filename := list[count];
      datefileptr^.isDirectory := false;
      DatesList.Add(datefileptr)
    end;
    errs.files := errs.files + 1;
  end;

  Result := errs;
end;


{*
 * Store dates in list object.
 * You MUST call InitDatesList first!
 * @param directory The directory path.
 }
function RememberRecursive(directory: String): MyErrors;
var
  SearchRec: TSearchRec;
  datefileptr: ^MyDateFile;
  timestamp: MyTimestamp;
  errs: MyErrors;
begin
  InitMyErrors(errs);
  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  try
    // Remember directory date.
    if GetMyTimestamp(directory, timestamp, true, errs) then begin
      New(datefileptr);
      datefileptr^.timestamp := timestamp;
      datefileptr^.filename := directory;
      datefileptr^.isDirectory := true;
      DatesList.Add(datefileptr);
    end;
    errs.dirs := errs.dirs + 1;
  finally
  end;


  if FindFirst(directory + '*.*', faDirectory+faReadOnly, SearchRec) = 0 then begin
    repeat
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        AddMyErrors(errs, RememberRecursive(directory+SearchRec.Name) );
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          // Remember file date.
          if GetMyTimestamp(directory+SearchRec.Name, timestamp, false, errs) then begin
            New(datefileptr);
            datefileptr^.timestamp := timestamp;
            datefileptr^.filename := directory+SearchRec.Name;
            datefileptr^.isDirectory := false;
            DatesList.Add(datefileptr);
          end;
          errs.files := errs.files + 1;
        end;
      end;
    until (not (FindNext(SearchRec) = 0));
    FindClose(SearchRec);
  end;

  Result := errs;
end;


{*
 * Save the remembered dates.
 * TODO Settings must be the same as when dates have been remembered.
 * TODO EXIF will be overwritten by zero if it was not present when remembered.
 * @return errors.
 *}
function SaveRemembered(): MyErrors;
var count: Integer;
    Attrs: Integer;
    errs: MyErrors;
    df: MyDateFile;
begin
  InitMyErrors(errs);

  if (DatesList<>nil) and (DatesList.Count<>0) then begin
    For count := 0 to DatesList.Count-1 do begin
      df := MyDateFile(DatesList.Items[count]^);
      if df.isDirectory then begin
        if Config.OptToolsSetDirTime then begin
          // set directory date
          errs.dirs := errs.dirs + 1;
          SetMyTimestamp(df.filename, df.timestamp, df.isDirectory, errs);
        end;
      end else begin
        // set file date
        errs.files := errs.files + 1;
        if not Unprotect(df.filename, Attrs) then begin
          errs.wprotected := errs.wprotected + 1
        end else begin
            SetMyTimestamp(df.filename, df.timestamp, df.isDirectory, errs);

            Protect(df.filename, Attrs);
        end;
      end;
    end;


  end;
  Result := errs;
end;


{*
 * Forget the remembered dates.
 *}
procedure ForgetRemembered();
var
  count: Integer;
begin
    if DatesList<>nil then begin
      try
        for count := 0 to DatesList.Count - 1 do begin
          Dispose(DatesList.Items[count]);
        end;
      finally
        DatesList.Clear;
      end;
    end;
end;

{*
 * Init MyTimestamp record with zeroes.
 * @param errs MyTimestamp record.
 *}
procedure InitTimestamp(var timestamp: MyTimestamp);
begin
  timestamp.created := 0;
  timestamp.modified := 0;
  timestamp.access := 0;
  timestamp.exifDT := 0;
  timestamp.exifDTO := 0;
  timestamp.exifDTD := 0;
end;

{*
 * Init MyErrors record with zeroes.
 * @param errs MyErrors record.
 *}
procedure InitMyErrors(var errs: MyErrors);
begin
  errs.files := 0;
  errs.dirs := 0;
  errs.wprotected := 0;
  errs.fileAccess := 0;
  errs.dirAccess := 0;
  errs.exifRead := 0;
  errs.exifWrite := 0;
end;

{*
 * Add MyErrors record to another one.
 * @param target MyErrors target record.
 * @param source MyErrors source record.
 *}
procedure AddMyErrors(var target: MyErrors; source: MyErrors);
begin
  target.files := target.files + source.files;
  target.dirs := target.dirs + source.dirs;
  target.wprotected := target.wprotected + source.wprotected;
  target.fileAccess := target.fileAccess + source.fileAccess;
  target.dirAccess := target.dirAccess + source.dirAccess;
  target.exifRead := target.exifRead + source.exifRead;
  target.exifWrite := target.exifWrite + source.exifWrite;
end;

function MaxMyErrors(errs1: MyErrors; errs2: MyErrors): MyErrors;
begin
  Result.files := Max(errs1.files, errs2.files);
  Result.dirs := Max(errs1.dirs, errs2.dirs);
  Result.wprotected := Max(errs1.wprotected, errs2.wprotected);
  Result.fileAccess := Max(errs1.fileAccess, errs2.fileAccess);
  Result.dirAccess := Max(errs1.dirAccess, errs2.dirAccess);
  Result.exifRead := Max(errs1.exifRead, errs2.exifRead);
  Result.exifWrite := Max(errs1.exifWrite, errs2.exifWrite);
end;

{*
 * Sets timestamp with selected tool options.
 * @param filename The file or directory path.
 * @param inc The increase time.
 * @param DateTime date and time to be set.
 * @param isDirectory true if directory, false if file
 * @param errs Returned errors.
 }
procedure SetMyTimestampToolOpts(filename: String; inc: MyIncTime; var DateTime: TDateTime; isDirectory: Boolean; var errs: MyErrors);
var
  timestamp: MyTimestamp;
  Attrs: Integer;
  myErrsGet: MyErrors;
  myErrsSet: MyErrors;
begin
  InitMyErrors(myErrsGet);
  InitMyErrors(myErrsSet);


  if isDirectory then begin

    // process directory

    //set directory date
    if GetMyTimestamp(filename, timestamp, true, myErrsGet) then begin
      if Config.OptOriginalTime = true then begin

        if Config.OptSource_EachTarget then begin

          if Config.OptTarget_Created
           then timestamp.created := IncDateTime(timestamp.created,inc);
          if Config.OptTarget_Modified
           then timestamp.modified := IncDateTime(timestamp.modified,inc);
          if Config.OptTarget_Access
           then timestamp.access := IncDateTime(timestamp.access,inc);

        end else begin
          if Config.OptSource_Created then DateTime := timestamp.created
          else if Config.OptSource_Modified then DateTime := timestamp.modified
          else if Config.OptSource_Access then DateTime := timestamp.access;
          DateTime := IncDateTime(DateTime,inc);

          if Config.OptTarget_Created then timestamp.created := DateTime;
          if Config.OptTarget_Modified then timestamp.modified := DateTime;
          if Config.OptTarget_Access then timestamp.access := DateTime;
        end;

        SetMyTimestamp(filename, timestamp, true, myErrsSet);
      end else begin
        if Config.OptTarget_Created then timestamp.created := DateTime;
        if Config.OptTarget_Modified then timestamp.modified := DateTime;
        if Config.OptTarget_Access then timestamp.access := DateTime;

        SetMyTimestamp(filename, timestamp, true, myErrsSet);

        DateTime := IncDateTime(DateTime,inc);
      end;

    end;


  end else begin

    // process file

    if not Unprotect(filename, Attrs) then begin
        errs.wprotected := errs.wprotected + 1;
    end else begin
      // TODO Really cancel if one file has no EXIF data? Or should we at least handle the other attributes?
      if GetMyTimestamp(filename, timestamp, false, myErrsGet) then begin
        if Config.OptOriginalTime {OptOriginalTime.checked = true} then begin
          if Config.OptSource_EachTarget then begin

            if Config.OptTarget_Created
             then timestamp.created := IncDateTime(timestamp.created,inc);
            if Config.OptTarget_Modified
             then timestamp.modified := IncDateTime(timestamp.modified,inc);
            if Config.OptTarget_Access
             then timestamp.access := IncDateTime(timestamp.access,inc);
            if Config.OptTarget_EXIF_DateTime
             then timestamp.exifDT := IncDateTime(timestamp.exifDT,inc);
            if Config.OptTarget_EXIF_DateTimeOriginal
             then timestamp.exifDTO := IncDateTime(timestamp.exifDTO,inc);
            if Config.OptTarget_EXIF_DateTimeDigitized
             then timestamp.exifDTD := IncDateTime(timestamp.exifDTD,inc);

          end else begin
            if Config.OptSource_Created then DateTime := timestamp.created
            else if Config.OptSource_Modified then DateTime := timestamp.modified
            else if Config.OptSource_Access then DateTime := timestamp.access
            else if Config.OptSource_EXIF_DateTime then DateTime := timestamp.exifDT
            else if Config.OptSource_EXIF_DateTimeOriginal then DateTime := timestamp.exifDTO
            else if Config.OptSource_EXIF_DateTimeDigitized then DateTime := timestamp.exifDTD;
            DateTime := IncDateTime(DateTime,inc);

            if Config.OptTarget_Created then timestamp.created := DateTime;
            if Config.OptTarget_Modified then timestamp.modified := DateTime;
            if Config.OptTarget_Access then timestamp.access := DateTime;
            if Config.OptTarget_EXIF_DateTime then timestamp.exifDT := DateTime;
            if Config.OptTarget_EXIF_DateTimeOriginal then timestamp.exifDTO := DateTime;
            if Config.OptTarget_EXIF_DateTimeDigitized then timestamp.exifDTD := DateTime;
          end;

          SetMyTimestamp(filename, timestamp, false, myErrsSet);
        end else begin
          if Config.OptTarget_Created then timestamp.created := DateTime;
          if Config.OptTarget_Modified then timestamp.modified := DateTime;
          if Config.OptTarget_Access then timestamp.access := DateTime;
          if Config.OptTarget_EXIF_DateTime then timestamp.exifDT := DateTime;
          if Config.OptTarget_EXIF_DateTimeOriginal then timestamp.exifDTO := DateTime;
          if Config.OptTarget_EXIF_DateTimeDigitized then timestamp.exifDTD := DateTime;

          SetMyTimestamp(filename, timestamp, false, myErrsSet);
          DateTime := IncDateTime(DateTime,inc);
        end;

        Protect(filename, Attrs);
      end;
    end;

  end;

  // Finally, sum up errors
  // (some errors may appear twice, once while getting and once while setting the timestamp)
  AddMyErrors(errs, MaxMyErrors(myErrsGet,myErrsSet) );  
end;


end.
