unit Config;

// This project is documented using the Doxygen documentation system.
// http://www.doxygen.org
// http://pas2dox.sourceforge.net

{* @file Config.pas
 * @brief Current configuration and string constants.
}

interface

uses
  Windows, SysUtils, Classes, IniFiles;

type
  {* @brief contains ALL available timestamps (c,m,a,exifDT,exifDTO,exifDTD) }
  MyTimestamp = record
    created, modified, access, exifDT, exifDTO, exifDTD: TDateTime;
  end;

  {* @brief contains booleans for ALL available timestamps (c,m,a,exifDT,exifDTO,exifDTD) }
  MyProcessed = record
    created, modified, access, exifDT, exifDTO, exifDTD: Boolean;
  end;

  {* @brief Increase time (secs, mins, hours, days, years) }
  MyIncTime = record
    s, m, h, d, a: Integer;
  end;

  {* @brief Number of errors }
  MyErrors = record
    files, dirs, wprotected, fileAccess, fileNoSource, dirAccess, exifRead, exifWrite: Integer;
  end;

  {* @brief Remembered date entry. }
  MyDateFile = record
    //date: Integer;
    timestamp: MyTimestamp;
    filename: String;
    isDirectory: boolean;
  end;

  MyUndo = record
    DatesList: TList;
    DatesListSettings: MyProcessed;
    singleTimestamp: TDateTime;
  end;

  function GetDefaultFtConversion(): String;


  function LoadSettingsNow(filename: String; isMainConfig: Boolean = false): Boolean;
  procedure SaveSettingsNow(filename: String; isMainConfig: Boolean = false);

  function GetIncTime(): MyIncTime;
  procedure SetIncTime(sec, min, hrs, ds, yrs: Integer);



const
  StrOptFtConversion_FileTimeToLocalFileTime: String = 'FileTimeToLocalFileTime';
  StrOptFtConversion_SystemTimeToTzSpecificLocalTime: String = 'SystemTimeToTzSpecificLocalTime';

  // const strings cannot be translated by dxgettext.
  StrIniFile: String = 'DDA.ini';
  StrIniSection: String = 'DDA Settings';
  StrIniFileExt: String = '.ini';
  StrCsvFileExt: String = '.csv';
  StrAboutUntranslated: String = '<AboutThisSoftware1.7.1>';
  // The following text is shown if there is no dxgettext translation
  // for '<AboutThisSoftware1.7.1>' - thus it should be written in English.
  StrAbout: String =
    '--------------------------------------------------'+#13#10
    +'Der Datei-Datums-nderer V1.7.1'+#13#10
    +'--------------------------------------------------'+#13#10
    +''+#13#10
    +' 2000-2016 by McSquirrel, Rapax, Rx Entertainment'+#13#10
    +''+#13#10
    +'Finally! A tool nobody should need, but everybody wants to have.'+#13#10
    +'With the Datei-Datums-nderer you can change the filedate of any file you '
    +'want. And you don''t have to pay for it because it is FREE!'+#13#10
    +''+#13#10
    +'Just open a file, change the date, and save the changings. That''s it.'+#13#10
    +''+#13#10
    +'Furthermore, since version 1.1, the Datei-Datums-nderer is able to '
    +'"remember" the filedates of a couple of files, so you can save them back '
    +'all at once. It''s fantastic!'+#13#10
    +''+#13#10
    +'From version 1.2 on, you can even save a new date and time to a couple of '
    +'files at once. If you want to, the time stamp is incremented file-by-file. '
    +'Version 1.3 even allows negative values, as well as increasing/decreasing '
    +'the original time stamp of multiple files. Using Version 1.4, you are able '
    +'to edit a directory''s time stamp or the whole contents of a directory '
    +'(recursively). Keyboard shortcuts provide you with quick access to all '
    +'functions. Wonderful!'+#13#10
    +''+#13#10
    +'Version 1.5 introduces an English user interface as well as full timestamp '
    +'support (created, modified, access) and EXIF support. Save your favourite '
    +'settings for easy restore at next startup.'+#13#10
    +''+#13#10
    +'Version 1.6 adds support for Windows 7 and newer Windows versions (since '
    +'Windows 7, the Windows Explorer and now also DDA use a new time function '
    +'for the correct handling of daylight saving time, as in former Windows '
    +'versions the DST file times were biased by one hour during standard time '
    +'and vice versa).'+#13#10
    +'Version 1.7 finally shows a progress window, so that long-taking actions '
    +'can be further examined, cancelled or even undone. Importing and exporting '
    +'CSV files is possible now. A new copy function allows for copying timestamps '
    +'recursively from one directory to another. Starting with Version 1.7.1, this '
    +'is also possible if the directory structure is different, only filename and '
    +'size must match.'+#13#10
    +''+#13#10
    +'(For further information, please switch to German and read the German manual.)'+#13#10
    +''+#13#10
    +''+#13#10
    +'Version history'+#13#10
    +'-----------------------'+#13#10
    +'v1.0 (04-Sep-2000): The Datei-Datums-nderer is born. You can change the '
    +'filedate of one file at once. There are bugs when altering '
    +'write-protected files. The info-window is of light blue. '+#13#10
    +''+#13#10
    +'v1.1 (07-Oct-2000): The bug while altering write-protected files has been '
    +'removed. You can tell the DDA how to handle these files.'+#13#10
    +''+#13#10
    +'Furthermore you can let the DDA "remember" the '
    +'filedates of several files located in one directory, '
    +'and save all the dates back by just a click. '+#13#10
    +''+#13#10
    +'v1.2 (27-Feb-2004): Now it''s possible to change the filedate of a couple of files in '
    +'one single step. If you want to, the time stamp is incremented for each '
    +'new file, so you can easily sort them by date then. '
    +'At startup, the input fields now contain actual date and time.'+#13#10
    +''+#13#10
    +'v1.3 (22-Oct-2007): As a new feature, DDA allows you to increase/decrease the original '
    +'time stamps of multiple files. The time difference input form now has '
    +'fields for seconds, minutes, hours, days and years. Oversized values '
    +'(such as 60 seconds) are converted automatically, negative values are '
    +'supported as well.'+#13#10
    +''+#13#10
    +'v1.4 (14-Mar-2009): Finally, DDA can handle directories recursively, and now even '
    +'directory time stamps are supported. Drag & drop allows for opening '
    +'single files quickly, while keyboard shortcuts give you access to all '
    +'DDA functions. The user interface now has the Windows XP look-and-feel. '
    +'Using input parameters, you may integrate DDA into Windows Explorer '
    +'(e.g. by using the "Open with..." menu).'+#13#10
    +''+#13#10
    +'v1.4.1 (18-Oct-2009): When increasing/decreasing time stamps, the sum of all increase '
    +'values (i.e. seconds, minutes etc.) needed to be different from 0 - this '
    +'has been fixed.'+#13#10
    +'Note: This version has been added to CHIP.de download archive on 29-Sep-2010.'+#13#10
    +''+#13#10
    +'v1.5 (11-Aug-2013): DDA now utilizes the Windows API to modify all available '
    +'timestamps (creation date, modified date, access date), you may choose the source '
    +'and target dates independly. Even EXIF data in .jpg files is supported by now '
    +'(based on dEXIF 1.03d, thanks to Gerry McGuire!), so you may correct the '
    +'timestamps of your digital photos, or copy the timestamps from EXIF to the '
    +'filesystem or vice versa.'+#13#10
    +'Date and time input has improved, you may also select to only change one of both, '
    +'and the problem loading/saving odd seconds has gone. All settings may be saved '
    +'as default or to separate settings files, so you may store different settings '
    +'for your common tasks.'+#13#10
    +'The software icon has been redrawn and is now available in a higher resolution. '
    +'Last but not least, the GUI may be switched from German to English (based on '
    +'GNU gettext / dxgettext), and the sourcecode has been cleaned up and documented '
    +'(doxygen).'+#13#10
    +''+#13#10
    +'v1.6 (01-Feb-2014): The edit menu got a new function which allows you to '
    +'set directory dates according to the latest file contained within.'+#13#10
    +'Advanced settings within the edit menu are now declared as such, and a '
    +'warning message will appear if you call any function which may alter '
    +'multiple files'+#13#10
    +'When increasing timestamps, a wizard now helps you to calculate the time '
    +'difference between the actual value and the desired one.'+#13#10
    +'Within the options menu, you may choose between different conversion '
    +'algorithms for file dates (starting from Windows 7, Windows Explorer uses '
    +'a new algorithm to convert file times to the local time zone, especially '
    +'respecting daylight saving time on NTFS formatted volumes).'+#13#10
    +'Under some circumstances, German users only got to see the English help text '
    +'if the language was set to "system default" - this has been fixed.'+#13#10
    +'Note: This version has been added to CHIP.de download archive on 09-Apr-2015, '
    +'and to GIGA.de download archive on 10-Mar-2016.'+#13#10
    +''+#13#10
    +'v1.7 (14-Aug-2016): When editing multiple files, you will now see a '
    +'progress window. This shows the current file in progress, as well as the '
    +'number of already processed files (and directories) and the current memory '
    +'usage. Long-lasting operations may be cancelled, and an undo function '
    +'allows you to reset the edited dates to the state before your edit.'+#13#10
    +'A user asked for the ability to set the directory date to the oldest file '
    +'contained within (alternatively to the newest file) - this lets you '
    +'reconstruct the creation date after copying files, for example.'+#13#10
    +'Another user suggested to add an export/import function for CSV files, '
    +'so you can easily edit data with external programs.'+#13#10
    +'However, there have also been some bugfixes: All recursive functions '
    +'may now also edit hidden or system files. The option to additionally set '
    +'the creation date for directories now only operates on directories (and not '
    +'files anymore). And when forgetting remembered file dates, the used memory '
    +'will now be freed completely (there was a memory leak before).'+#13#10
    +'Note: This version has been added to GIGA.de download archive on '
    +'17-Aug-2016.'+#13#10
    +''+#13#10
    +'v1.7.1 (19-Jul-2022): Up to now, it was only possible to recursively '
    +'copy the file dates from one directory to another if both directories '
    +'had the same directory structure. On demand of a user, a new function '
    +'was added which does this regardless of the directory structure, comparing '
    +'only the filename and file size.'+#13#10
    +''+#13#10
    +'DDA.exe has been built by the free "Borland Delphi 6 Personal Edition" '
    +'and then has been compressed using "UPX" (the "Ultimate '
    +'Packer for eXecutables").'+#13#10
    +''+#13#10
    +''+#13#10
    +'Future'+#13#10
    +'----------'+#13#10
    +'The main window shall become more intuitive. A file list may let you '
    +'add files and directories easily, sort them according to your needs, and '
    +'then process them in the given order.'+#13#10
    +''+#13#10
    +'An integrated file system browser might help you navigating through '
    +'directories (without changing the "last access" timestamp) and view all '
    +'relevant data at a glance.'+#13#10
    +''+#13#10
    +'A demo mode may let you try out some settings before actually applying the '
    +'new dates to your files.'+#13#10
    +''+#13#10
    +''+#13#10
    +'Author'+#13#10
    +'----------'+#13#10
    +'DDA was written by:'+#13#10
    +''+#13#10
    +'Manuel Haim'+#13#10
    +'Schuetzenstrasse 19'+#13#10
    +'35104 Lichtenfels-Muenden'+#13#10
    +'GERMANY'+#13#10
    +''+#13#10
    +'Email: McSquirrel@gmx.de'+#13#10
    ;

  // For convenience, the German translation is embedded here:
  StrAboutDE: String =
    '--------------------------------------------------'+#13#10
    +'Der Datei-Datums-nderer V1.7.1'+#13#10
    +'--------------------------------------------------'+#13#10
    +''+#13#10
    +' 2000-2022 by McSquirrel, Rapax, Rx Entertainment'+#13#10
    +''+#13#10
    +'Na endlich! Ein Tool, das keiner brauchen sollte und doch jeder haben will:'+#13#10
    +'Mit dem Datei-Datums-nderer knnen Sie nach Belieben das nderungsdatum '
    +'und weitere Zeitstempel Ihrer Dateien und Verzeichnisse ndern, anpassen, '
    +'exportieren und vieles mehr! Und Sie brauchen noch nicht einmal einen '
    +'Pfennig zu bezahlen, denn der Datei-Datums-nderer ist freie Software!'+#13#10
    +''+#13#10
    +'ffnen Sie eine beliebige Datei, ndern Sie das Datum, und speichern Sie sie '
    +'wieder. Fertig.'+#13#10
    +''+#13#10
    +'Des weiteren merkt sich der Datei-Datums-nderer ab Version 1.1 auf '
    +'Knopfdruck das Datum von beliebig vielen Dateien in einem Verzeichnis '
    +'und speichert es auf Knopfdruck wieder zurck. Fantastisch!'+#13#10
    +''+#13#10
    +'Ab Version 1.2 knnen Sie auch eine Reihe von Dateien einheitlich mit einem '
    +'neuen Datum versehen. Datum eintragen, Dateien whlen, fertig. Auf Wunsch '
    +'wird fr jede Datei der Zeitstempel erhht. Negative Werte sind ab Version '
    +'1.3 mglich, ebenso das Erhhen/Erniedrigen des Original-Zeitstempels '
    +'mehrerer Dateien. Ab Version 1.4 kann auch das Verzeichnis-Datum oder ein '
    +'komplettes Verzeichnis mitsamt Inhalt (rekursiv) bearbeitet werden - '
    +'Tastenkrzel machen die Verarbeitung noch einfacher. Wunderbar!'+#13#10
    +''+#13#10
    +'Sie mchten mehr aus Ihren Dateien herausholen, neben dem nderungs- auch '
    +'das Erstellungs- und Zugriffsdatum anpassen oder den Zeitstempel mit den '
    +'EXIF-Bilddaten abgleichen? Ab Version 1.5 kein Problem mehr, denn Quell- '
    +'und Zieldatum sind bei allen Aktionen frei whlbar. All ihre Einstellungen '
    +'knnen Sie selbstverstndlich auch speichern. Klasse!'+#13#10
    +''+#13#10
    +'Version 1.6 untersttzt nun auch Windows 7 und neuere Windows-Versionen '
    +'(ab Windows 7 wird im Windows-Explorer und nun auch in DDA zum korrekten '
    +'Umgang mit Sommer- und Winterzeit eine neue Zeitfunktion verwendet, der '
    +'Windows-Explorer in frheren Windows-Versionen hatte das Dateidatum nach '
    +'einer Zeitumstellung um eine Stunde verschoben angezeigt).'+#13#10
    +''+#13#10
    +'Version 1.7 zeigt bei der Bearbeitung endlich ein Fortschrittsfenster, '
    +'lngere Aktionen werden somit nachvollziehbarer, lassen sich abbrechen '
    +'oder sogar rckgngig machen. Auch ein CSV-Import und -Export ist jetzt '
    +'mglich. Neu ist auerdem eine Kopierfunktion, um Zeitstempel rekursiv '
    +'von einem auf ein anderes Verzeichnis zu bertragen. Ab Version 1.7.1 '
    +'klappt dies auch bei abweichender Verzeichnisstruktur, solange Dateiname '
    +'und Dateigre bereinstimmen.'+#13#10
    +''+#13#10
    +''+#13#10
    +'Anwendungsbeispiele'+#13#10
    +'----------------------------------'+#13#10
    +'Bei kleinen nderungen an Dateien mchte man ggf. das ursprngliche '
    +'nderungsdatum beibehalten. Dies ist z.B. bei MP3-Dateien der Fall, wenn '
    +'man nur die ID3-Tags ndert. Die Merk-Funktion von DDA ist hier hilfreich:'+#13#10
    +''+#13#10
    +'- Bearbeiten-Men => Daten von mehreren Dateien / Verzeichnis-Inhalt merken'+#13#10
    +'- dann die Dateien mit der jeweiligen Software bearbeiten'+#13#10
    +'- Bearbeiten-Men => Daten von mehreren Dateien zurckschreiben'+#13#10
    +'- Bearbeiten-Men => Im Speicher befindliche Daten lschen'+#13#10
    +''+#13#10
    +''+#13#10
    +'Bei der Entwicklung von Digitalfotos liefern manche Anbieter die Abzge nach '
    +'Dateidatum sortiert. Hat man am ein oder anderen Bild aber nderungen '
    +'vorgenommen, so bringt dies die Reihenfolge durcheinander. Die Mglichkeit, '
    +'den Dateien wieder einheitliche Zeitstempel zu geben und von Datei zu Datei '
    +'zu erhhen, ist mit DDA gegeben:'+#13#10
    +''+#13#10
    +'- Start-Datum/Uhrzeit im Hauptfenster eintragen'+#13#10
    +'- Bearbeiten-Men => dabei Zeitstempel erhhen => Zeitdifferenz eintragen'+#13#10
    +'- Bearbeiten-Men => Datum fr mehrere Dateien setzen'+#13#10
    +''+#13#10
    +''+#13#10
    +'Oder, falls die Bilder noch EXIF-Daten enthalten, kann auch der Zeitstempel '
    +'daraus extrahiert und in die Dateiattribute bertragen werden:'+#13#10
    +''+#13#10
    +'- Optionen-Men => Quelldatum: EXIF Date Time Original'+#13#10
    +'- Optionen-Men => Zieldatum: Gendert am'+#13#10
    +'- Bearbeiten-Men => dabei Original-Dateidatum als Basis nutzen'+#13#10
    +'- Bearbeiten-Men => Datum fr mehrere Dateien setzen'+#13#10
    +''+#13#10
    +''+#13#10
    +'Manche Autoradios, DVD- oder MP3-Player sortieren selbst heute noch die '
    +'Dateien nach ihrem nderungsdatum. Mit DDA knnen leicht alle Dateien und '
    +'Unterverzeichnisse in alphabetischer Reihenfolge mit aufsteigendem Datum '
    +'versehen werden, so dass dem erwarteten Hrgenuss nichts mehr im Wege '
    +'steht:'+#13#10
    +''+#13#10
    +'- Start-Datum/Uhrzeit im Hauptfenster eintragen'+#13#10
    +'- ggf. Optionen-Men => Experten-Einstellungen anpassen'+#13#10
    +'- Bearbeiten-Men => dabei Zeitstempel erhhen => Zeitdifferenz eintragen'+#13#10
    +'- Bearbeiten-Men => Datum fr Verzeichnis-Inhalt rekursiv setzen'+#13#10
    +''+#13#10
    +'Interessant sind in diesem Zusammenhang auch die Optionen "Erst Dateien, '
    +'dann Unterverzeichnisse" (sorgt dafr, dass zunchst alle Dateien innerhalb '
    +'eines Verzeichnisses in alphabetischer Reihenfolge bearbeitet werden und '
    +'danach mit den enthaltenen Unterverzeichnissen in alphabetischer Reihenfolge '
    +'fortgefahren wird) sowie "Breitensuche statt Tiefensuche" (hierbei werden '
    +'zunchst alle Verzeichnisse der obersten Ebene abgearbeitet, dann alle '
    +'Verzeichnisse der nchsttieferen Ebene usw.).'+#13#10
    +''+#13#10
    +''+#13#10
    +'War die Uhr einer Digitalkamera falsch gestellt, so haben auch die Fotodateien '
    +'ein falsches nderungsdatum. Dies lsst sich mit DDA beheben, indem das '
    +'Original-Dateidatum als Basis genutzt und um eine frei whlbare Zeit nach vorn '
    +'oder hinten verschoben wird:'+#13#10
    +''+#13#10
    +'- Bearbeiten-Men => dabei Zeitstempel erhhen => Zeitdifferenz eintragen'+#13#10
    +'- Bearbeiten-Men => dabei Original-Dateidatum als Basis nutzen'+#13#10
    +'- Bearbeiten-Men => Datum fr mehrere Dateien setzen'+#13#10
    +''+#13#10
    +''+#13#10
    +'Bekannte Fehler und Probleme'+#13#10
    +'-------------------------------------------------'+#13#10
    +'Was wre ein Windows-Proggi ohne Fehler? Naja, wohl irgendwie nicht so '
    +'ganz dasselbe...'+#13#10
    +''+#13#10
    +'Zunchst: Nicht jedes beliebige Datum lsst sich speichern. So werden Sie z.B. '
    +'darauf verzichten mssen, den MP3-Dateien Ihrer Oldies-Sammlung oder Ihren '
    +'eingescannten Familienfotos authentische Daten zu verleihen, nur um sie im '
    +'Windows-Explorer entsprechend nach Erscheinungsjahr zu sortieren. Damals '
    +'gab es noch keine Heimcomputer, und das nderungsdatum ist eben nur im '
    +'Kontext der Erzeugung oder nderung von neuen Dateien innerhalb '
    +'informationsverarbeitender Systeme erfunden worden, nicht aber, um nach '
    +'vllig freiem Belieben gendert zu werden. Zur Kontrolle wird nach dem '
    +'Speichern einer Datei das neu vergebene Datum oberhalb der Eingabefelder '
    +'angezeigt.'+#13#10
    +''+#13#10
    +'blicherweise erscheint jedesmal eine Fehlermeldung, wenn versucht wurde, '
    +'eine Datei zu ffnen bzw. zu schreiben, die gerade von einem anderen '
    +'Programm benutzt wird oder ausgefhrt wird. Manchmal ist eine Datei noch '
    +'auf diese Weise gesperrt, obwohl sie bereits geschlossen wurde. Dann kann '
    +'es helfen, die jeweiligen Programme zu beenden oder sogar den PC neu zu '
    +'starten.'+#13#10
    +''+#13#10
    +'Das Datum des letzten Zugriffs auf eine Datei kann leicht zerstrt werden, '
    +'insbesondere wenn im Windows Explorer beim Navigieren durch Verzeichnisse '
    +'weitere Dateieigenschaften ausgelesen werden (z.B. EXIF-Daten oder ein '
    +'Vorschau-Bild).'+#13#10
    +''+#13#10
    +'Das ndern des Verzeichnisdatums wird von Windows 95, 98 und ME nicht '
    +'untersttzt - hier wren manuelle Eingriffe ins Dateisystem oder die Entwicklung '
    +'eines speziellen Treibers erforderlich. Kleine Programmfehler knnten dabei '
    +'erheblichen Datenverlust zur Folge haben. Mit DDA und einem aktuellen '
    +'Windows-Betriebssystem sind und bleiben Sie jedoch auf der sicheren Seite.'+#13#10
    +''+#13#10
    +'Bei der Mehrfach-Dateiauswahl wird die zuletzt ausgewhlte Datei zuerst '
    +'bearbeitet (das liegt am Windows-Dateifenster), also muss man ggf. die Dateien '
    +'von unten nach oben markieren. Beim Markieren aller Dateien mit Strg-A oder '
    +'dem Maus-Lasso tritt das Problem nicht auf.'+#13#10
    +''+#13#10
    +'Als Sekunden knnen ggf. nur gerade Zahlenwerte gespeichert werden (das '
    +'liegt aber am Dateisystem von Windows).'+#13#10
    +''+#13#10
    +'Wenn eigene Dateien auf der Festplatte pltzlich ein anderes Datum zeigen als '
    +'eine Kopie auf CD-ROM oder in einem ZIP-Archiv, hngt dies mit der Zeitzonen- '
    +'oder Jahreszeit-Einstellung in Windows zusammen. In den NT-basierten Windows-'
    +'Versionen bis einschlielich Windows Vista wird bei NTFS-formatierten Datentrgern '
    +'das Datum vom Windows Explorer u.a. abhngig von Sommer- oder '
    +'Winterzeit dargestellt, d.h. im Winter ist die Zeit von Sommer-Dateien um '
    +'eine Stunde verschoben und umgekehrt. Beispiel: Wer an einem Sommertag um 16:00 '
    +'Uhr ein Bild speichert, der wird im Winter bei derselben Datei 15:00 Uhr lesen '
    +'drfen (vgl. Microsoft KB 129574 - zum Ausprobieren einfach mal das Systemdatum '
    +'ndern und die Ansicht im Windows Explorer aktualisieren). Ziemlich verwirrend, '
    +'insbesondere bei Digitalfotos...'+#13#10
    +'Unter Windows 7 wurde dieser Schwachpunkt bzgl. NTFS-formatierten Datentrgern '
    +'behoben, dahingegen ergibt sich das umgekehrte (!) Problem bzgl. Nicht-NTFS-formatierten '
    +'Datentrgern (wie z.B. FAT-formatierten USB-Sticks oder Speicherkarten): '
    +'Eine zur Sommerzeit um 16:00 Uhr gespeicherte Datei zeigt im Winter 17:00 Uhr.'+#13#10
    +'Seit DDA 1.6 lsst sich der gewnschte Algorithmus im Optionen-Men whlen.'+#13#10
    +''+#13#10
    +''+#13#10
    +'Versionsgeschichte'+#13#10
    +'-------------------------------'+#13#10
    +'v1.0 (04-Sep-2000): Der Datei-Datums-nderer erblickt das Licht der Welt. '
    +'Von jeweils einer Datei lsst sich das nderungsdatum ndern. Beim Umgang '
    +'mit schreibgeschtzten Dateien treten Probleme auf. Das Info-Fenster ist '
    +'hellblau.'+#13#10
    +''+#13#10
    +'v1.1 (07-Okt-2000): Das Problem mit den schreibgeschtzten Dateien wurde '
    +'behoben. ber einen neuen Menpunkt bestimmt der Anwender, wie der DDA '
    +'mit solchen Dateien verfahren soll.'+#13#10
    +'Dieses Info-Fenster ist knigsblau und hat eine Scroolbar bekommen --------------->'+#13#10
    +''+#13#10
    +'Auerdem besteht nun die Mglichkeit, das Datum von mehreren sich in einem '
    +'Verzeichnis befindenden Dateien zu "merken" und auf Knopfdruck wieder '
    +'zurckspeichern.'+#13#10
    +''+#13#10
    +'v1.2 (27-Feb-2004): Auf Anregung eines Benutzers ist es nun mglich, '
    +'mehreren Dateien gleichzeitig ein Datum zuzuweisen. Wahlweise kann der '
    +'Zeitstempel pro Datei automatisch erhht werden, so dass z.B. eine Reihe '
    +'von Dateien aufsteigend nach Alphabet sortiert unterschiedliche Daten '
    +'zugewiesen bekommt.'+#13#10
    +'Weitere Neuigkeit: Die Eingabefelder des DDA enthalten beim Programmstart '
    +'nun schon das aktuelle Datum sowie die Uhrzeit.'+#13#10
    +''+#13#10
    +'v1.3 (22-Okt-2007): Ein weiterer Benutzer uerte den Wunsch, die Original-'
    +'Zeitstempel von mehreren Dateien zu erhhen. Dies kann nun getan werden. '
    +'Auerdem erfolgt die Eingabe der Zeitstempel-Differenz nun nach Sekunden, '
    +'Minuten, Stunden, Tagen und Jahren gegliedert. bergroe Werte (wie z.B. '
    +'60 Sekunden) werden automatisch umgerechnet, erstmals sind auch negative '
    +'Werte erlaubt.'+#13#10
    +''+#13#10
    +'v1.4 (14-Mr-2009): Dank der Anregung eines Benutzers lsst sich jetzt ein '
    +'Verzeichnis komplett mit allen Unterverzeichnissen (rekursiv) bearbeiten. '
    +'Neue Tastenkrzel ermglichen einen schnellen Zugriff auf alle Funktionen (die '
    +'aktivierten Optionen werden direkt im Hauptfenster angezeigt). Einzelne Dateien '
    +'und Verzeichnisse knnen auch auf das Programmfenster gezogen und dort '
    +'fallen gelassen werden, um sie zur Bearbeitung zu ffnen (Drag & Drop).'+#13#10
    +'Das Erscheinungsbild wurde an Windows XP angepasst und leicht berarbeitet. '
    +'Auerdem untersttzt DDA erstmals Eingabeparameter, so dass eine Integration '
    +'in den Windows Explorer (z.B. ber das "ffnen mit..."-Men) mglich ist.'+#13#10
    +''+#13#10
    +'v1.4.1 (18-Okt-2009): Ein Erhhen der Zeitstempel war nicht mglich, wenn die '
    +'Werte fr Sekunden, Minuten usw. in der Summe 0 ergaben - dieser Fehler '
    +'wurde behoben.'+#13#10
    +'Hinweis: Diese Version wurde am 29-Sep-2010 ins Download-Archiv von CHIP.de aufgenommen!'+#13#10
    +''+#13#10
    +'v1.5 (11-Aug-2013): Viele schon lange geplante, teils durch Benutzer '
    +'angeregte nderungen sind in diese Version eingeflossen. Durch die direkte '
    +'Verwendung von Windows-Betriebssystem-Funktionen knnen nun endlich alle '
    +'Dateisystem-Zeitstempel (Erstelldatum, nderungsdatum, Zugriffsdatum) '
    +'flexibel angepasst werden, Quelle und Ziel sind hierbei frei whlbar.'+#13#10
    +'Eine Untersttzung fr EXIF-Zeitstempel in .jpg-Dateien (basierend auf '
    +'dEXIF 1.03d von Gerry McGuire, vielen Dank!) ermglicht eine einfache '
    +'Korrektur des Datums von Digitalfotos oder auch ein Umkopieren zwischen '
    +'Dateisystem- und EXIF-Zeitstempeln.'+#13#10
    +'Neben einer verbesserten Zeit- und Datumseingabe lsst sich jetzt wahlweise '
    +'auch nur die Uhrzeit oder nur das Datum ndern, auerdem werden ungerade '
    +'Sekunden verarbeitet. Alle vorgenommenen Einstellungen lassen sich global '
    +'oder in separaten Dateien speichern, so dass der Benutzer fr spezielle '
    +'Aufgaben eigene Konfigurationsdateien anlegen kann.'+#13#10
    +'Das Programm-Icon wurde komplett neu gezeichnet und liegt jetzt in hherer '
    +'Auflsung vor. '
    +'Zuguterletzt ist die Benutzeroberflche nun wahlweise auf deutsch oder '
    +'englisch anzusehen (basierend auf GNU gettext / dxgettext), und der '
    +'Quellcode wurde weiter bereinigt sowie dokumentiert (doxygen).'+#13#10
    +''+#13#10
    +'v1.6 (01-Feb-2014): Das Bearbeiten-Men hat auf Anregung eines Benutzers eine '
    +'neue Funktion erhalten, mit der sich das Datum von Verzeichnissen auf die '
    +'neueste darin enthaltene Datei setzen lsst. '
    +'Nebenbei wurden im Bearbeiten-Men die erweiterten Einstellungen als solche '
    +'gekennzeichnet und ein Warnhinweis bei allen Funktionen eingebaut, '
    +'die mehrere Dateien gleichzeitig verndern.'+#13#10
    +'Bei der Funktion "Zeitstempel erhhen" erleichtert ein '
    +'Differenz-Taschenrechner unter Angabe von Ist- und Soll-Zeitstempel nun '
    +'die Anpassung der Zeitstempel von '
    +'mehreren Dateien, z.B. wenn die Uhr der Digitalkamera falsch gestellt war.'+#13#10
    +'Im Optionen-Men lsst sich der zu verwendende Konvertierungsalgorithmus fr '
    +'Datei-Zeitstempel whlen (ab Windows 7 rechnet der Windows Explorer die '
    +'Dateizeit anders in lokale Zeit um als bei vorigen Windows-Versionen, um '
    +'insbesondere Probleme mit der Winter- und Sommerzeitumstellung zu beheben).'+#13#10
    +'Statt des deutschen Hilfetextes wurde ggf. der englische angezeigt, wenn '
    +'"Systemvorgabe" als Sprache ausgewhlt war - dies wurde korrigiert.'+#13#10
    +'Hinweis: Diese Version wurde am 09-Apr-2015 ins Download-Archiv von CHIP.de '
    +'und am 10-Mr-2016 ins Download-Archiv von GIGA.de aufgenommen!'+#13#10
    +''+#13#10
    +'v1.7 (14-Aug-2016): Bei der Verarbeitung mehrerer Dateien friert DDA nun '
    +'nicht mehr ein, sondern zeigt eine interaktive Fortschrittsanzeige. Neben der '
    +'aktuell verarbeiteten Datei wird darin die Anzahl der bereits erledigten '
    +'Dateien (und Verzeichnisse) sowie der aktuelle Speicherverbrauch angezeigt. '
    +'Auerdem lassen sich lnger andauernde Aktionen bei Bedarf einfach abbrechen.'+#13#10
    +'Darberhinaus ermglicht eine Rckgngig-Funktion das Zurcksetzen der '
    +'genderten Daten auf den Stand vor der letzten Bearbeitung.'+#13#10
    +'Auf Wunsch eines Benutzers ist es jetzt mglich, das Datum von Verzeichnissen '
    +'alternativ zur neuesten auch auf die lteste darin enthaltene Datei zu '
    +'setzen - so lsst sich z.B. das Erstelldatum von Verzeichnissen nach einem '
    +'Kopiervorgang rekonstruieren.'+#13#10
    +'Ein weiterer Benutzer hat eine Export- und Import-Funktion fr Dateidaten '
    +'(als CSV-Datei) angeregt, um Daten einfach mit externen Programmen '
    +'weiterverarbeiten zu knnen.'+#13#10
    +'Neben den genannten neuen Funktionen gab es aber auch Fehlerbereinigungen: '
    +'Die rekursiven Funktionen bercksichtigen inzwischen auch versteckte '
    +'Dateien sowie Systemdateien. Auerdem wurde ein Fehler bei der Option '
    +'"bei Verzeichnissen zustzlich Erstelldatum setzen" behoben (zuvor wurden '
    +'hierbei offenbar auch Dateien entsprechend bearbeitet).'+#13#10
    +'Beim Merken von Dateidaten wurde ein Speicherleck behoben, der belegte '
    +'Hauptspeicher wird jetzt nach dem Lschen wieder vollstndig freigegeben.'+#13#10
    +'Hinweis: Diese Version wurde am 17-Aug-2016 ins Download-Archiv von GIGA.de '
    +'aufgenommen!'+#13#10
    +''+#13#10
    +'v1.7.1 (19-Jul-2022): Bislang konnte das Dateidatum nur rekursiv von einem '
    +'auf ein anderes Verzeichnis bertragen werden, wenn beide Verzeichnisse '
    +'dieselbe Verzeichnisstruktur aufwiesen. Auf Wunsch eines Benutzers ist '
    +'dies nun mit einer neuen Funktion auch bei abweichender Verzeichnisstruktur '
    +'mglich, es mssen lediglich Dateiname und Dateigre bereinstimmen.'+#13#10
    +''+#13#10
    +'Die DDA.exe wurde mit der kostenlosen "Borland Delphi 6 Personal Edition" '
    +'kompiliert und wurde nachtrglich mit "UPX v3.03" (einem Packer fr '
    +'ausfhrbare Dateien) gepackt.'+#13#10
    +''+#13#10
    +''+#13#10
    +'Planung und Ideen'+#13#10
    +'------------------------------'+#13#10
    +'Das Hauptfenster soll bersichtlicher und die Bedienung insgesamt '
    +'intuitiver werden. ber eine Dateiliste knnten Dateien und Verzeichnisse '
    +'zur Bearbeitung hinzugefgt und dann der Reihenfolge nach bearbeitet werden, '
    +'auf Wunsch auch rekursiv. Als Quelle oder Ziel knnte auch der Dateiname '
    +'oder Daten aus einer CSV-Datei dienen. Die Zuordnung Quelle-Ziel knnte '
    +'ber ein ansprechendes Auswahlfenster erfolgen.'+#13#10
    +''+#13#10
    +'Ein eigener Verzeichnis-Browser knnte dabei helfen, durch Verzeichnisse '
    +'zu navigieren, ohne das "Letzer Zugriff"-Datum zu ndern, und die '
    +'Dateidaten tabellarisch anzeigen. Auch eine Suche nach Dateien '
    +'gem bestimmten Kriterien wre denkbar.'+#13#10
    +''+#13#10
    +'Ein Demo-Modus sowie eine Rckgngig-Funktion wrden Einsteigern helfen, '
    +'nderungen zunchst auszuprobieren und dann durchzufhren.'+#13#10
    +''+#13#10
    +''+#13#10
    +'Programmautor'+#13#10
    +'------------------------'+#13#10
    +'Mich knnen Sie unter folgender Adresse erreichen:'+#13#10
    +''+#13#10
    +'Manuel Haim'+#13#10
    +'Schtzenstrae 19'+#13#10
    +'35104 Lichtenfels-Mnden'+#13#10
    +''+#13#10
    +'E-Mail: McSquirrel@gmx.de'+#13#10
    ;

resourcestring
  // These strings can be translated by dxgettext.
  AskWhichFile =
    'Open date from file...';
  AskWhichDir =
    'Open date from directory...';
  AskRememberWhichFiles =
    'Which files'' dates shall be remembered?';
  AskSetWhichFiles =
    'Which files'' dates shall be set?';
  AskIniWhichFile =
    'Load settings file...';
  AskSaveIniWhichFile =
    'Save settings file...';
  AskRememberWhichDir =
    'Which directory''s dates shall be remembered recursively?';
  AskSetWhichDir =
    'Which directory''s dates shall be set recursively?';
  AskSaveCsv =
    'Save CSV file...';
  AskSaveCsvWhichDir =
    'Which directory''s dates shall be saved to CSV file recursively?';
  AskLoadCsv =
    'Open CSV file...';
  AskSaveSettings =
    'Disable auto-save of settings permanently?'+#13#10
    +'(If you click "Yes", the file "%s%" will be deleted.)';
  AskReallyUndo =
    'Undo %n% changes, processed on %t% ?';
  AskDisableUndo =
    'Undo has been disabled. Shall the current undo buffer be cleared as well?';
  AskCopyFromDir =
    'Which directory''s file dates shall be copied recursively?';
  AskCopyToDir =
    'To which directory shall the dates be copied?';
  AskCopyNameSizeFromDir =
    'Which directory''s file dates shall be copied by filename/size?';
  ErrDate =
    'The date you entered is invalid!'+#13#10
    +'Thus, the current date will be used.';
  ErrOpenFile =
    'An unexpected error occured when opening the file.'+#13#10
    +'Maybe the file is running or in use by another program.'
    +' Please close the file or program and try again.';
  ErrOpenFileNoSource =
    'Unspecified source date.'+#13#10
    +'When editing a single file or directory, you must always specify a single source date.'+#13#10
    +'Please select a source date option different from'
    +' "Each selected target date"!';
  ErrOpenFileEXIF =
    'Cannot read EXIF data.'+#13#10
    +'Maybe the file does not contain any EXIF data,'
    +' or the file is in use by another program.'
    +' Please close the file or program and try again.';
  ErrOpenDragDrop =
    'The file could not be opened.';
  ErrSaveWriteProtect =
    'Cannot bypass the file''s write protection.'
    +' Please make sure the medium is not write-protected.';
  ErrSaveWriteProtect2 =
    'The file is write protected. Please enable the option'
    +' "Process write-protected files" in the Options menu'
    +' and try again!';
  ErrSaveWriteProtect3 =
    'Cannot save the file. Please make sure the medium is'
    +' not write-protected (e.g. CD-ROM?) !';
  ErrSaveWriteProtectEXIF =
    'Cannot save the file. Please note that EXIF data can only'
    +' be written to JPG files which already contain EXIF data.'
    +' Please also make sure the medium is'
    +' not write-protected (e.g. CD-ROM?) !';
  ErrSaveDir =
    'Cannot save the directory date. Please make sure the medium is'
    +' not write-protected (e.g. CD-ROM?) !';
  ErrSaveNone =
    'Please open a file or directory first.';
  ErrOpenDir =
    'An unexpected error occured when opening the directory.'
    +' Maybe the directory or its date are not accessible at this time.';
  ErrLoadIni =
    'Invalid settings file!'+#13#10
    +'Settings have been reset to default values.';
  MsgFileExists = 'File exists, overwrite?';
  //MsgSaveDates =
  //  'Please make sure that you did not change source/target date options'
  //  +' before saving remembered dates, as this may lead to unpredictable'
  //  +' results!';
  MsgForgetDates =
    'This will remove all remembered dates from memory.' + #13#10
    +' Afterwards, these dates cannot be written back to their files'
    +' anymore.';
  //MsgRememberRecursiveSourceTargetWarning =
  //  'When remembering or saving remembered dates, source and target date options must be identical!' + #13#10
  //  +'Please select "Each selected target date" or "Same as source date" first!';
  MsgRememberRecursiveWarning =
    'Remembering a directory''s dates recursively (incl. all dates from all'
    +' files and subdirectories contained therein) may take some time.';
  MsgSetMultipleWarning =
    'You are going to change the dates of multiple files.';
  MsgSetRecursiveWarning =
    'Changing a directory''s dates recursively (incl. all dates from all'
    +' files and subdirectories contained therein) may take some time.';
  MsgSetDirDatesLatestSourceTargetWarning =
   'When setting directory dates based on the newest or oldest file contained within,'
   +' only a single source date may be selected.' +#13#10
   +'Please choose a source date different from "Each selected target date"!';
  MsgSetDirDatesLatestRecursiveWarning =
    'Changing a directory''s dates recursively (crawling through all'
    +' files and subdirectories contained therein) may take some time.';
  MsgReadSourceFiles =
    'Source files are being read...';
  MsgWriteTargetFiles =
    'Target files are being written...';

  StrNoFile = 'no file opened - drag file here';
  StrFile = 'File:';
  StrDir = 'Directory:';
  StrSourceVal = 'Original value:';
  StrOptions = 'Options:';
  StrOptionsNone = 'Options: none';

  StrTargetDate = 'Target date:';
  StrOptCreated = '- Created';
  StrOptModified = '- Modified';
  StrOptLastAccess = '- Last access';
  StrOptExifDT = '- EXIF Date Time';
  StrOptExifDTO = '- EXIF Date Time Original';
  StrOptExifDTD = '- EXIF Date Time Digitized';

  ErrRemoveProtection1 = 'One time, the write-protection could not be disabled'
                             +' (medium was write-protected).';
  ErrRemoveProtectionN = '%n% times, the write-protection could not be disabled'
                              +' (medium was write-protected).';
  ErrProtected1 = 'One file was write-protected.';
  ErrProtectedN = '%n% files were write-protected.';
  ErrFileAccess1 = 'One file date could not be accessed (maybe in use by another program,'
                  +' stored on a write-protected medium or invalid filename).';
  ErrFileAccessN = '%n% file dates could not be accessed (maybe in use by another program,'
                   +' stored on a write-protected medium or invalid filename).';
  ErrFileNoSource1 = 'One file had no matching source file to copy from.';
  ErrFileNoSourceN = '%n% files had no matching source file to copy from.';
  ErrDirAccess1 = 'One directory date could not be accessed (maybe in use by another program,'
                  +' stored on a write-protected medium or invalid filename).';
  ErrDirAccessN = '%n% directory dates could not be accessed (maybe in use by another program,'
                  +' stored on a write-protected medium or invalid filename).';
  ErrNoExif1 = 'One file had no EXIF data.';
  ErrNoExifN = '%n% files had no EXIF data.';
  MsgDirs1 = 'one directory';
  MsgDirsN = '%n% directories';
  MsgFiles1 = 'one file';
  MsgFilesN = '%n% files';
  MsgDirsFiles = '%files% and %dirs% have been processed without errors.';
  ErrDirsFiles = '%files% and %dirs% have been processed, but with errors:';

  MsgDirsFilesProc = '%n% files + %m% dirs processed.';
  MsgTimeElapsed = 'Time elapsed: %s%.';
  MsgMemUsage = 'Memory usage: %s%.';
  MsgProcStarted = 'Process started.';
  MsgProcFinished = 'Process finished.';
  MsgProcCancelled = 'Process cancelled.';

  MsgAdjustOptions = 'For processing, the options will be adjusted as follows:';
  ErrCsvImportHeaders = 'The first line of your CSV file contains invalid or '
  + 'incomplete headers. Valid headers are:';
  ErrCsvFileEmpty = 'The CSV file is empty!';
  MsgCsvReadErr = 'Error reading CSV file!';
  MsgCsvWriteErr = 'Error writing CSV file!';

  MenuOptMainTime =
    '- based on main window date';
  MenuOptMainTime_Disabled =
    '- based on main window date (disabled)';
  MenuOptOriginalTime =
    '- based on original file date';
  MenuOptOriginalTime_Disabled =
    '- based on original file date (disabled)';
  MenuOptIncreaseTime =
    '- increase timestamp (%time% per file)';
  MenuOptIncreaseTime_Disabled =
    '- increase timestamp per file (disabled)';
  MenuOptMainTime2 =
    '- if no file present, set main window date';
  MenuOptMainTime2_Disabled =
    '- if no file present, set main window date (disabled)';
  MenuOptLatestSubdir =
    '- also consider subdirs when looking for newest/oldest file';
  MenuOptLatestSubdir_Disabled =
    '- also consider subdirs when looking for newest/oldest file (disabled)';

  MenuOptFtConversion_DEFAULT =
    '- System default (%default%)'; 

  StrAboutTranslated = '<AboutThisSoftware1.7.1>'; // should be replaced by gnugettext
  MsgConfigFileVersionWarning =
    'Caution! Your settings file was created by a different DDA version!'
    +' Please consider updating DDA.'+#13#10
    +#13#10
    +'However, the file was loaded, but may lead to unexpected results'
    +' in this version of DDA. You have been warned.';

  StrCancel = 'Cancel';
  StrClose = 'Close';
  StrReallyCancel = 'Really cancel?';

  StrProgressCaption = 'Progress...';

  StrColMandatory = '(mandatory)';

var
  //FileHandle: Integer;
  OpenedFile, OpenedDir: boolean;
  CurrentFileName, CurrentFileVersion:String;
  CurrentFiletime: TDateTime;
  CurrentFiletimeAvail: boolean;
  CurrentSettings:TMemIniFile;
  DatesList: TList; // list of MyDateFile
  DatesListSettings: MyProcessed;
  HashList: THashedStringList; // hash of String, MyDateFile
  UndoList: TList; // list of MyUndo
  CurrentIncTime:MyIncTime;

  curProcErrors: MyErrors;
  curProcCancelled: boolean;
  curProcFile: String;
  curProcFileErrors: MyErrors;

  CurrentFtConversion: String;

  OptTime,
  OptDate,
  OptSource_EachTarget,
  OptSource_Created,
  OptSource_Modified,
  OptSource_Access,
  OptSource_EXIF_DateTime,
  OptSource_EXIF_DateTimeOriginal,
  OptSource_EXIF_DateTimeDigitized,
  OptTarget_Source,
  OptTarget_Created,
  OptTarget_Modified,
  OptTarget_Access,
  OptTarget_EXIF_DateTime,
  OptTarget_EXIF_DateTimeOriginal,
  OptTarget_EXIF_DateTimeDigitized,
  OptHidden,
  OptSystem,
  OptWriteProtected,
  OptUndo,
  OptErrDetails,
  OptLogDetails,
  OptSetDirCreateTime: Boolean;
  OptFtConversion: String;
  OptToolsSetDirTime,
  OptFilesFirst,
  OptBreadthFirst,
  OptOriginalTime: Boolean;
  OptMainTime2: Boolean;
  OptLatestSubdir: Boolean;

  OptLanguage: String;


  DefaultFtConversion : String = '';
type
  //http://msdn2.microsoft.com/en-us/library/ms724833.aspx
  TOSVersionInfoEx = packed record
    dwOSVersionInfoSize : DWORD;
    dwMajorVersion      : DWORD;
    dwMinorVersion      : DWORD;
    dwBuildNumber       : DWORD;
    dwPlatformId        : DWORD;
    szCSDVersion        : array[0..127] of Char;
    wServicePackMajor   : WORD;
    wServicePackMinor   : WORD;
    wSuiteMask          : WORD;
    wProductType        : BYTE;
    wReserved           : BYTE;
  end;

const
  VER_SUITE_PERSONAL  = $00000200;
  VER_NT_WORKSTATION  = $00000001;
  VER_SUITE_WH_SERVER = $00008000;
  SM_SERVERR2         = 89;

function GetOSVersionEx(var lpVersionInformation: TOSVersionInfoEx): BOOL; stdcall; external kernel32 name
'GetVersionExA';

function GetOSVersionInfoEx : TOSVersionInfoEx;


implementation

{*
 * Get Windows version number.
 * Thanks to http://www.delphi-treff.de/tipps/system/systemeigenschaften/windows-version-ermitteln/
 * @return version.
 }
function GetOSVersionInfoEx : TOSVersionInfoEx;
var
  OSVersionInfo   : TOSVersionInfo absolute Result;
  Done : Boolean;
begin
  FillChar(Result, SizeOf(Result), #0);
  Done := False;
  try
    Result.dwOSVersionInfoSize := SizeOf(TOSVersionInfoEx);
    Done := GetOSVersionEx(Result);
  except
  end;
  if not(Done) then
  begin
    try
      FillChar(Result, SizeOf(Result), #0);
      Result.dwOSVersionInfoSize := SizeOf(TOSVersionInfo);
      Done := GetVersionEx(OSVersionInfo);
    except
    end;
  end;
end;

function GetDefaultFtConversion(): String;
var
  osversion: TOSVersionInfoEx;
begin
  if DefaultFtConversion = '' then begin
    osversion := GetOSVersionInfoEx();
    if (osversion.dwMajorVersion >= 7) or ((osversion.dwMajorVersion = 6) and (osversion.dwMinorVersion >= 1)) then begin
      // Windows 7 and above
      DefaultFtConversion := Config.StrOptFtConversion_SystemTimeToTzSpecificLocalTime;
    end else begin
      // Windows 95 thru Windows Vista
      DefaultFtConversion := Config.StrOptFtConversion_FileTimeToLocalFileTime;
    end;
  end;
  Result := DefaultFtConversion;
end;

{*
 * Load settings from file. On errors, default values are used.
 * @param filename The path to the ini file.
 * @param isMainConfig true if main config is loaded, false if custom config.
 * @return true if valid ini file, false otherwise.
 }
function LoadSettingsNow(filename: String; isMainConfig: Boolean = false): Boolean;
begin
  Result := true;
  CurrentSettings.Rename(filename,true);

  if FileExists(filename) and not CurrentSettings.SectionExists(StrIniSection) then Result := false;

  CurrentFileVersion := CurrentSettings.ReadString(StrIniSection,'Version','');

  OptTime := CurrentSettings.ReadBool(StrIniSection,'OptTime',true);
  OptDate := CurrentSettings.ReadBool(StrIniSection,'OptDate',true);
  OptSource_EachTarget := CurrentSettings.ReadBool(StrIniSection,'OptSource_EachTarget',false);
  OptSource_Created := CurrentSettings.ReadBool(StrIniSection,'OptSource_Created',false);
  OptSource_Modified := CurrentSettings.ReadBool(StrIniSection,'OptSource_Modified',true);
  OptSource_Access := CurrentSettings.ReadBool(StrIniSection,'OptSource_Access',false);
  OptSource_EXIF_DateTime := CurrentSettings.ReadBool(StrIniSection,'OptSource_EXIF_DateTime',false);
  OptSource_EXIF_DateTimeOriginal := CurrentSettings.ReadBool(StrIniSection,'OptSource_EXIF_DateTimeOriginal',false);
  OptSource_EXIF_DateTimeDigitized := CurrentSettings.ReadBool(StrIniSection,'OptSource_EXIF_DateTimeDigitized',false);
  OptTarget_Source := CurrentSettings.ReadBool(StrIniSection,'OptTarget_Source',true);
  OptTarget_Created := CurrentSettings.ReadBool(StrIniSection,'OptTarget_Created',false);
  OptTarget_Modified := CurrentSettings.ReadBool(StrIniSection,'OptTarget_Modified',false);
  OptTarget_Access := CurrentSettings.ReadBool(StrIniSection,'OptTarget_Access',false);
  OptTarget_EXIF_DateTime := CurrentSettings.ReadBool(StrIniSection,'OptTarget_EXIF_DateTime',false);
  OptTarget_EXIF_DateTimeOriginal := CurrentSettings.ReadBool(StrIniSection,'OptTarget_EXIF_DateTimeOriginal',false);
  OptTarget_EXIF_DateTimeDigitized := CurrentSettings.ReadBool(StrIniSection,'OptTarget_EXIF_DateTimeDigitized',false);
  if OptTarget_Source then begin
    OptTarget_Created := OptSource_Created;
    OptTarget_Modified := OptSource_Modified;
    OptTarget_Access := OptSource_Access;
    OptTarget_EXIF_DateTime := OptSource_EXIF_DateTime;
    OptTarget_EXIF_DateTimeOriginal := OptSource_EXIF_DateTimeOriginal;
    OptTarget_EXIF_DateTimeDigitized := OptSource_EXIF_DateTimeDigitized;
  end;
  // TODO: Validate that there is only one source selected.
  OptHidden := CurrentSettings.ReadBool(StrIniSection,'OptHidden',true);
  OptSystem := CurrentSettings.ReadBool(StrIniSection,'OptSystem',false);
  OptWriteProtected := CurrentSettings.ReadBool(StrIniSection,'OptWriteProtected',false);
  OptUndo := CurrentSettings.ReadBool(StrIniSection,'OptUndo',true);
  OptErrDetails := CurrentSettings.ReadBool(StrIniSection,'OptErrDetails',true);
  OptLogDetails := CurrentSettings.ReadBool(StrIniSection,'OptLogDetails',false);
  OptSetDirCreateTime := CurrentSettings.ReadBool(StrIniSection,'OptSetDirCreateTime',false);
  OptFtConversion := CurrentSettings.ReadString(StrIniSection,'OptFtConversion','default');
  if ( OptFtConversion='default' ) then CurrentFtConversion := GetDefaultFtConversion();
  OptToolsSetDirTime := CurrentSettings.ReadBool(StrIniSection,'OptToolsSetDirTime',false);
  OptFilesFirst := CurrentSettings.ReadBool(StrIniSection,'OptFilesFirst',false);
  OptBreadthFirst := CurrentSettings.ReadBool(StrIniSection,'OptBreadthFirst',false);

  OptOriginalTime := CurrentSettings.ReadBool(StrIniSection,'OptOriginalTime',false);

  SetIncTime(
    CurrentSettings.ReadInteger(StrIniSection,'OptIncreaseTimeS',0),
    CurrentSettings.ReadInteger(StrIniSection,'OptIncreaseTimeM',0),
    CurrentSettings.ReadInteger(StrIniSection,'OptIncreaseTimeH',0),
    CurrentSettings.ReadInteger(StrIniSection,'OptIncreaseTimeD',0),
    CurrentSettings.ReadInteger(StrIniSection,'OptIncreaseTimeA',0)
  );

  OptMainTime2 := CurrentSettings.ReadBool(StrIniSection,'OptMainTime2',false);
  OptLatestSubdir := CurrentSettings.ReadBool(StrIniSection,'OptLatestSubdir',false);

  // language will only be loaded from main configuration file
  if isMainConfig then OptLanguage := CurrentSettings.ReadString(StrIniSection,'OptLanguage','default');
end;

{* Save settings to file.
 * @param filename The path to the ini file.
 * @param isMainConfig true if main config is saved, false if custom config.
 }
procedure SaveSettingsNow(filename: String; isMainConfig: Boolean = false);
begin
  CurrentSettings.Rename(filename,false);
  CurrentSettings.WriteString(StrIniSection,'Version','1.5');

  //Do not save this here...
  //CurrentSettings.WriteBool(StrIniSection,'OptSaveSettings',OptSaveSettings.Checked);

  CurrentSettings.WriteBool(StrIniSection,'OptTime',OptTime);
  CurrentSettings.WriteBool(StrIniSection,'OptDate',OptDate);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_EachTarget',OptSource_EachTarget);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_Created',OptSource_Created);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_Modified',OptSource_Modified);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_Access',OptSource_Access);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_EXIF_DateTime',OptSource_EXIF_DateTime);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_EXIF_DateTimeOriginal',OptSource_EXIF_DateTimeOriginal);
  CurrentSettings.WriteBool(StrIniSection,'OptSource_EXIF_DateTimeDigitized',OptSource_EXIF_DateTimeDigitized);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_Source',OptTarget_Source);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_Created',OptTarget_Created);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_Modified',OptTarget_Modified);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_Access',OptTarget_Access);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_EXIF_DateTime',OptTarget_EXIF_DateTime);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_EXIF_DateTimeOriginal',OptTarget_EXIF_DateTimeOriginal);
  CurrentSettings.WriteBool(StrIniSection,'OptTarget_EXIF_DateTimeDigitized',OptTarget_EXIF_DateTimeDigitized);
  CurrentSettings.WriteBool(StrIniSection,'OptHidden',OptHidden);
  CurrentSettings.WriteBool(StrIniSection,'OptSystem',OptSystem);
  CurrentSettings.WriteBool(StrIniSection,'OptWriteProtected',OptWriteProtected);
  CurrentSettings.WriteBool(StrIniSection,'OptUndo',OptUndo);
  CurrentSettings.WriteBool(StrIniSection,'OptErrDetails',OptErrDetails);
  CurrentSettings.WriteBool(StrIniSection,'OptLogDetails',OptLogDetails);
  CurrentSettings.WriteBool(StrIniSection,'OptSetDirCreateTime',OptSetDirCreateTime);
  CurrentSettings.WriteString(StrIniSection,'OptFtConversion',OptFtConversion);
  CurrentSettings.WriteBool(StrIniSection,'OptToolsSetDirTime',OptToolsSetDirTime);
  CurrentSettings.WriteBool(StrIniSection,'OptFilesFirst',OptFilesFirst);
  CurrentSettings.WriteBool(StrIniSection,'OptBreadthFirst',OptBreadthFirst);

  CurrentSettings.WriteBool(StrIniSection,'OptOriginalTime',OptOriginalTime);

  CurrentSettings.WriteInteger(StrIniSection,'OptIncreaseTimeS',CurrentIncTime.s);
  CurrentSettings.WriteInteger(StrIniSection,'OptIncreaseTimeM',CurrentIncTime.m);
  CurrentSettings.WriteInteger(StrIniSection,'OptIncreaseTimeH',CurrentIncTime.h);
  CurrentSettings.WriteInteger(StrIniSection,'OptIncreaseTimeD',CurrentIncTime.d);
  CurrentSettings.WriteInteger(StrIniSection,'OptIncreaseTimeA',CurrentIncTime.a);

  CurrentSettings.WriteBool(StrIniSection,'OptMainTime2',OptMainTime2);
  CurrentSettings.WriteBool(StrIniSection,'OptLatestSubdir',OptLatestSubdir);

  // language will only be saved in main configuration file
  if isMainConfig then CurrentSettings. WriteString(StrIniSection,'OptLanguage',OptLanguage);

  try
    CurrentSettings.UpdateFile;
  finally
  end;
//
end;

{*
 * Get data from the increase time form.
 }
function GetIncTime(): MyIncTime;
begin
  Result := CurrentIncTime;
end;

{*
 * Set increase time.
 * @param sec seconds
 * @param min minutes
 * @param hrs hours
 * @param ds days
 * @param yrs years
 }
procedure SetIncTime(sec, min, hrs, ds, yrs: Integer);
begin
  CurrentIncTime.s:=sec;
  CurrentIncTime.m:=min;
  CurrentIncTime.h:=hrs;
  CurrentIncTime.d:=ds;
  CurrentIncTime.a:=yrs;
end;


end.
