unit Helpers;

// This project is documented using the Doxygen documentation system.
// http://www.doxygen.org
// http://pas2dox.sourceforge.net

{* @file Helpers.pas
 * @brief Various helper functions.
}

interface

uses
  Windows, DateUtils, SysUtils, Classes, Contnrs, Math, Forms, Dialogs, Controls, IniFiles,
  Config, dEXIF, Unit4;


    // Helper functions for conversions and write-protection
    function NumStri(var arg: String): Integer;
    function Stri(number: Integer): String;
    function ReplaceString(arg, oldString, newString: String): String;
    function DTToFT(const dtLocal: TDateTime): TFileTime;
    function FTToDT(const ft: TFileTime): TDateTime;
    function Unprotect(filename: String; var Attrs: Integer): Boolean;
    procedure Protect(filename: String; Attrs: Integer);

    // created/modified/accessed timestamp from file system
    function GetFsTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean): Boolean;
    function SetFsTimestamp(filename: String; timestamp: MyTimestamp; isDirectory, byConfig: Boolean): Boolean;

    procedure UpdateDT(var oldDateTime: TDateTime; newDateTime: TDateTime);
    //procedure UpdateMT(var oldMyTimestamp: MyTimestamp; newMyTimestamp: MyTimestamp);

    // get/set a complete timestamp with all values which are selected in options
    function GetMyTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;
    function SetMyTimestamp(filename: String; timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;

    // called by Unit1 on single files/dirs
    function getFileDateTime(filename: String; var DateTime: TDateTime; init: Boolean): Boolean;
    function getDirDateTime(filename: String; var DateTime: TDateTime; init: Boolean): Boolean;
    function setFileDateTime(filename: String; DateTime: TDateTime; init: Boolean): Boolean;
    function setDirDateTime(filename: String; DateTime: TDateTime; init: Boolean): Boolean;

    function IncDateTime(DateTime: TDateTime; inc: MyIncTime): TDateTime;

    function SetList(list: TStrings; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
    function SetRecursiveBF(directory: String; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
    function SetRecursive(directory: String; inc: MyIncTime; var DateTime: TDateTime; init: Boolean): MyErrors;

    function CopyRecursive(directory, directory1, directory2: String; init: Boolean): MyErrors;

    function CopyNameSize(directory1, directory2: String): MyErrors;
    function CopyNameSizeReadRecursive(directory: String): MyErrors;
    function CopyNameSizeWriteRecursive(directory: String): MyErrors;

    function SetDirRecursiveLatest(directory: String; MainDateTime: TDateTime; var extremum: TDateTime; newest: Boolean; init: Boolean): MyErrors;

    procedure InitDatesList();
    procedure DestroyDatesList();

    procedure InitHashList();
    procedure DestroyHashList();

    procedure InitUndoList();
    procedure DestroyUndoList();

    function RememberList(list: TStrings): MyErrors;
    function RememberRecursive(directory: String; init: Boolean): MyErrors;
    function SaveRemembered(): MyErrors;
    function ExportRememberedCsv(filename: String): MyErrors;
    procedure ForgetRemembered();

    function SaveCsv(directory, filename: String): MyErrors;
    procedure SaveCsvRecursive(directory: String; var f1: TextFile);
    function ImportCsv(filename: String): MyErrors;

    procedure InitTimestamp(var timestamp: MyTimestamp);
    procedure InitMyErrors(var errs: MyErrors);
    procedure InitCurProcErrors();
    procedure AddMyErrors(var target: MyErrors; source: MyErrors);
    function MaxMyErrors(errs1: MyErrors; errs2: MyErrors): MyErrors;
    procedure SetMyTimestampToolOpts(filename: String; inc: MyIncTime; var DateTime: TDateTime; isDirectory: Boolean);

    function MyStringToDateTime(dtstring: String): TDateTime;

    procedure InitUndo();
    procedure AddUndo(filename: String; timestamp: MyTimestamp; isDirectory: boolean);
    function Undo(): MyErrors;
    function HasUndo(): Boolean;

    function TzSpecificLocalTimeToSystemTime(lpTimeZoneInformation: PTimeZoneInformation;
      var lpLocalTime, lpUniversalTime: TSystemTime): BOOL; stdcall;
    {$EXTERNALSYM TzSpecificLocalTimeToSystemTime}
    function LocalDateTimeToUTC(aLocal : TDateTime) : TDateTime;

const
  fdDefault: Integer = 0;

implementation


{*
 * Validate String number (remove non-number chars).
 * @param arg A string containing a number.
 * @return The number (0 if String was empty).
 }
function NumStri(var arg: String): Integer;
var
  code, number:Integer;
begin
  if arg='' then begin
    number := 0;
  end else begin
    VAL(arg, number, code);
    Str(number, arg);
  end;
  Result := number;
end;

{*
 * Convert Integer to String.
 * @param number A number.
 * @return The number as string.
 }
function Stri(number: Integer): String;
var arg: string;
begin
  Str(number, arg);
  Stri := arg;
end;

{*
 * Replace all occurrences of a substring.
 * @param arg The string.
 * @param oldString The substring to be replaced.
 * @param newString The replacement substring.
 }
function ReplaceString(arg, oldString, newString: String): String;
var
  tmp: String;
begin
  tmp := '';

  while Pos(oldString, arg) > 0 do begin
    tmp := tmp + copy(arg, 1, Pos(oldString, arg)-1);
    tmp := tmp + newString;
    Delete(arg, 1, Pos(oldString, arg)-1 + Length(oldString));
  end;

  Result := tmp + arg;
end;


// Note: The following comment is OBSOLETE, problem has been resolved by using
// specific funtions...
//
// Problems with daylight savings time under Win7 (1h bias),
// i.e. DDA does not see the same time as Windows Explorer under Win7.
//
// Tried to implement the following solution:
// Well... as all available solutions do not lead to a common result anyway,
// we could just use FindFirstFile for reading file dates (as Windows Explorer
// sees them), and have another look after saving a file to find the filesystem's
// bias (then saving the file again!)
//
//
// Maybe see http://www.codeproject.com/Articles/1144/Beating-the-Daylight-Savings-Time-bug-and-getting
// http://www.administrator.de/forum/sommerzeit-winterzeit-v%C3%B6llig-falsche-zeit-ntfs-zu-fat-bzw-fat32-unter-xp-84580.html
//
// Tz patch only helps for Win7 NTFS, but not FAT (why?)
//
// FindFirstFile could maybe read the time as shown in Windows Explorer,
// but we also need a way to save the date (how should we do this?)
//
// (non-DST = no Daylight Savings Time = Winter = GMT+1)
// (DST = Daylight Savings Time = Summer = GMT+2)
//
// Winter.txt on FAT-formatted USB stick
// -----------------------------------------------
// WinXP non-DST (Winter):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 10:00 AM
//   DDA 1.5:           01-Jan-2014 10:00 AM
// WinXP DST (Summer):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 10:00 AM
//   DDA 1.5:           01-Jan-2014 10:00 AM
// Win7 non-DST (Winter):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 10:00 AM
//   DDA 1.5:           01-Jan-2014 10:00 AM
//   DDA TzPatch:       01-Jan-2014 10:00 AM
// Win7 DST (Summer):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 09:00 AM
//   DDA 1.5:           01-Jan-2014 11:00 AM
//   DDA TzPatch:       01-Jan-2014 10:00 AM well...
//
// Summer.txt on FAT-formatted USB stick
// -----------------------------------------------
// WinXP non-DST (Winter):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 10:00 AM
//   DDA 1.5:           01-Jul-2014 10:00 AM
//   DDA TzPatch:       01-Jul-2014 ??:00 AM
// WinXP DST (Summer):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 10:00 AM
//   DDA 1.5:           01-Jul-2014 10:00 AM
//   DDA TzPatch:       01-Jul-2014 ??:00 AM
// Win7 non-DST (Winter):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 11:00 AM
//   DDA 1.5:           01-Jul-2014 10:00 AM
//   DDA TzPatch:       01-Jul-2014 11:00 AM good!
// Win7 DST (Summer):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 10:00 AM
//   DDA 1.5:           01-Jul-2014 11:00 AM
//   DDA TzPatch:       01-Jul-2014 11:00 AM well...
//
// Winter.txt on NTFS-formatted USB stick
// -----------------------------------------------
// WinXP non-DST (Winter):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 10:00 AM
//   DDA 1.5:           01-Jan-2014 10:00 AM
//   DDA TzPatch:       01-Jan-2014 ??:00 AM
// WinXP DST (Summer):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 11:00 AM
//   DDA 1.5:           01-Jan-2014 11:00 AM
//   DDA TzPatch:       01-Jan-2014 ??:00 AM
// Win7 non-DST (Winter):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 10:00 AM
//   DDA 1.5:           01-Jan-2014 10:00 AM
//   DDA TzPatch:       01-Jan-2014 10:00 AM
// Win7 DST (Summer):
//   GetFileTime:       01-Jan-2014 09:00 AM
//   Windows Explorer:  01-Jan-2014 10:00 AM
//   DDA 1.5:           01-Jan-2014 11:00 AM
//   DDA TzPatch:       01-Jan-2014 10:00 AM good!
//
// Summer.txt on NTFS-formatted USB stick
// -----------------------------------------------
// WinXP non-DST (Winter):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 10:00 AM
//   DDA 1.5:           01-Jul-2014 10:00 AM
//   DDA TzPatch:       01-Jul-2014 ??:00 AM
// WinXP DST (Summer):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 11:00 AM
//   DDA 1.5:           01-Jul-2014 11:00 AM
//   DDA TzPatch:       01-Jul-2014 ??:00 AM
// Win7 non-DST (Winter):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 11:00 AM
//   DDA 1.5:           01-Jul-2014 10:00 AM
//   DDA TzPatch:       01-Jul-2014 11:00 AM good!
// Win7 DST (Summer):
//   GetFileTime:       01-Jul-2014 09:00 AM
//   Windows Explorer:  01-Jul-2014 11:00 AM
//   DDA 1.5:           01-Jul-2014 11:00 AM
//   DDA TzPatch:       01-Jul-2014 11:00 AM
//


// SystemTimeToTzSpecificLocalTime / TzSpecificLocalTimeToSystemTime works for NTFS on Win7,
// but is not available for Win9x/NT/2000, shows other dates on WinXP,
// and produces some weird problems for Non-NTFS (e.g. FAT) on Win7


// Functions re-programmed for Win9x/NT/2000
// Thanks to http://www.delphipraxis.net/44052-tzspecificlocaltimetosystemtime-fuer-win9x-nt-2000-a.html
// - Well, EncodeDateTime throws an error under Win7, thus we skip this...
{
function GetDateTimeForBiasSystemTime(GivenDateTime: TSystemTime; GivenYear: integer): TDateTime;
var
  Year, Month, Day: word;
  Hour, Minute, Second, MilliSecond: word;
begin
  GivenDateTime.wYear := GivenYear;
  while not TryEncodeDayOfWeekInMonth(
   GivenDateTime.wYear, GivenDateTime.wMonth, GivenDateTime.wDay,
   GivenDateTime.wDayOfWeek, Result)
  do begin
    Dec(GivenDateTime.wDay);
  end;

  DecodeDateTime(Result, Year, Month, Day, Hour, Minute, Second, MilliSecond);
  Result := EncodeDateTime(Year, Month, Day, GivenDateTime.wHour,
   GivenDateTime.wMinute, GivenDateTime.wSecond, GivenDateTime.wMilliseconds);
end;

function GetBiasForDate(GivenDateTime: TDateTime): integer;
var
  tzi: TIME_ZONE_INFORMATION;
begin
  GetTimeZoneInformation(tzi);
  if (GivenDateTime < GetDateTimeForBiasSystemTime(tzi.StandardDate, YearOf(GivenDateTime)))
  and (GivenDateTime >= GetDateTimeForBiasSystemTime(tzi.DaylightDate, YearOf(GivenDateTime)))
  then begin
    Result := (tzi.Bias + tzi.DaylightBias) * (-1);
  end else begin
    Result := (tzi.Bias + tzi.StandardBias) * (-1);
  end;
end;

function UTCToLocalDateTime2(aUTC: TDateTime): TDateTime;
begin
  Result := IncMinute(aUTC, GetBiasForDate(aUTC));
end;

function LocalDateTimeToUTC2(aLocal: TDateTime): TDateTime;
begin
  Result := IncMinute(aLocal, GetBiasForDate(aLocal) * -1);
end;
}


// Following functions only available on Windows XP and later systems
//Thanks to http://www.delphipraxis.net/42772-referenz-zeitzone-ohne-sommer-winterzeit.html
// TODO 27-Jul-2016 function seems obsolete
{
function UtcTOLocalDateTime (aUTC : TDateTime) : TDateTime;
var
  //tzi : TIME_ZONE_INFORMATION;
  utc : TSystemTime;
  localtime : TSystemTime;
begin
  DateTimeToSystemTime(aUTC,utc);
  //GetTimeZoneInformation(tzi);
  //SystemTimeToTzSpecificLocalTime(@tzi,utc,localtime);
  SystemTimeToTzSpecificLocalTime(nil,utc,localtime);
  Result := SystemTimeToDateTime(localtime);
end;
}

function TzSpecificLocalTimeToSystemTime; external kernel32 name 'TzSpecificLocalTimeToSystemTime';


function LocalDateTimeToUTC(aLocal : TDateTime) : TDateTime;
var
  //tzi : TIME_ZONE_INFORMATION;
  utc : TSystemTime;
  localtime : TSystemTime;
begin
  DateTimeToSystemTime(aLocal, localtime);
  //GetTimeZoneInformation(tzi);
  //TzSpecificLocalTimeToSystemTime(@tzi, localtime, utc);
  TzSpecificLocalTimeToSystemTime(nil, localtime, utc);
  Result := SystemTimeToDateTime(utc);
end;

{*
 * Convert TDateTime to TFileTime.
 * Thanks to
 * https://forums.codegear.com/thread.jspa?threadID=11125&tstart=0
 * @param dtLocal The time as TDateTime.
 * @return The time as TFileTime.
 }
function DTToFT(const dtLocal: TDateTime): TFileTime;
var
ft: TFileTime;
st: TSystemTime;
stLocal: TSystemTime;
begin
  if (Config.CurrentFtConversion = Config.StrOptFtConversion_SystemTimeToTzSpecificLocalTime) then begin
    // New way - Windows 7 NTFS and later (user should choose this in options menu):
    DateTimeToSystemTime(dtLocal, stLocal);
    TzSpecificLocalTimeToSystemTime(nil, stLocal, st);
    SystemTimeToFileTime(st, ft);
    Result := ft;
  end else begin
    // Old way - Windows XP and before:
    DateTimeToSystemTime(dtLocal, st);
    SystemTimeToFileTime(st, ft);
    LocalFileTimeToFileTime(ft, ft);
    Result := ft;
  end;
end;

{*
 * Convert TFileTime to TDateTime.
 * @param filetime The time as TFileTime.
 * @return The time as TDateTime.
 }
function FTToDT(const ft: TFileTime): TDateTime;
var
ftLocal: TFileTime;
stLocal: TSystemTime;
st: TSystemTime;
dtLocal: TDateTime;
begin
  if (Config.CurrentFtConversion = Config.StrOptFtConversion_SystemTimeToTzSpecificLocalTime) then begin
    // New way - Windows 7 NTFS and later (user should choose this in options menu):
    FileTimeToSystemTime(ft, st);
    SystemTimeToTzSpecificLocalTime(nil,st,stLocal);
    Result := SystemTimeToDateTime(stLocal);
  end else begin
    // Old way - Windows XP and before:
    FileTimeToLocalFileTime(ft, ftLocal);
    FileTimeToSystemTime(ftLocal, stLocal);
    dtLocal := SystemTimeToDateTime(stLocal);
    Result := dtLocal;
  end;
end;

{*
 * Remove write protection.
 * @param filename The file path.
 * @param Attrs The original file attributes.
 * @return true if unprotected, false otherwise.
 }
function Unprotect(filename: String; var Attrs: Integer): Boolean;
var err: Integer;
begin
  Result := true;
  Attrs := FileGetAttr(filename);
  if not (Attrs and faReadOnly = 0) then begin
    if Config.OptWriteProtected then begin
      err := FileSetAttr(filename, Attrs - faReadOnly);
      if not err=0 then Result := false;
    end else
      Result := false;
  end;
end;

{*
 * Restore write protection.
 * @param filename The file path.
 * @param Attrs The original file attributes.
 }
procedure Protect(filename: String; Attrs: Integer);
begin
  if (Attrs and faReadOnly <> 0) and (Config.OptWriteProtected) then begin
      FileSetAttr(filename, Attrs);
  end;
end;

{*
 * Get a file's / directory's timestamp from file system.
 * (Windows 95/98/ME: No directories supported!)
 }
function GetFsTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean): Boolean;
var F: THandle;
  AgeCreated, AgeAccess, AgeModified: TFileTime;
begin
  Result := False;

  // get file handle
  if isDirectory then begin
    F := CreateFileA(PAnsiChar(FileName), GENERIC_READ,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, 0);
  end else begin //isFile
    F := CreateFileA(PAnsiChar(filename), GENERIC_READ,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, 0, 0);
  end;
  if F = INVALID_HANDLE_VALUE then Exit;

  // get file time
  Result := GetFileTime(F, @AgeCreated, @AgeAccess, @AgeModified);

  timestamp.created := FTToDT(AgeCreated);
  timestamp.modified := FTToDT(AgeModified);
  timestamp.access := FTToDT(AgeAccess);

  CloseHandle(F);
end;

{*
 * Set a file's / directory's timestamp in the file system.
 * (Windows 95/98/ME: No directories supported!)
 * If param byConfig is set, the particular data will be set depending on:
 * - Config.OptTarget_Created
 * - Config.OptTarget_Modified
 * - Config.OptTarget_Access
 *
 * @param filename The path.
 * @param timestamp The timestamp.
 * @param isDirectory true if directory, false if file.
 * @param byConfig true to save only activated datetimes,
 *   false to save complete timestamp.
 * @return true if successful, false otherwise.
 }
function SetFsTimestamp(filename: String; timestamp: MyTimestamp; isDirectory, byConfig: Boolean): Boolean;
var F: THandle;
  AgeCreated, AgeAccess, AgeModified: TFileTime;
begin
  Result := false;

  // get file handle
  if isDirectory then begin
    F := CreateFileA(PAnsiChar(FileName), GENERIC_WRITE,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, 0);
  end else begin //isFile
    F := CreateFileA(PAnsiChar(FileName), GENERIC_WRITE,
      FILE_SHARE_READ or FILE_SHARE_WRITE or FILE_SHARE_DELETE, nil,
      OPEN_EXISTING, 0, 0);
  end;
  if F = INVALID_HANDLE_VALUE then Exit;

  AgeCreated := DTToFT(timestamp.created);
  AgeAccess := DTToFT(timestamp.access);
  AgeModified := DTToFT(timestamp.modified);

  // set file time
  if byConfig then begin
    Result := true;
    if Config.OptTarget_Created then {CreateTime} Result := Result and SetFileTime(F, @AgeCreated, nil, nil);
    if Config.OptTarget_Access then {LastAccessTime} Result := Result and SetFileTime(F, nil, @AgeAccess, nil);
    if Config.OptTarget_Modified then {LastWriteTime}  Result := Result and SetFileTime(F, nil, nil, @AgeModified);
  end else begin
    Result := SetFileTime(F, @AgeCreated, @AgeAccess, @AgeModified);
  end;

  CloseHandle(F);
end;

{*
 * Updates date and/or time depending on the user's choice.
 * (Config.OptDate and Config.OptTime are set by Checkbox_Date and Checkbox_Time)
 * @param oldDateTime The datetime to be updated.
 * @param newDateTime The new datetime to be set.
 }
procedure UpdateDT(var oldDateTime: TDateTime; newDateTime: TDateTime);
var dt: TDateTime;
begin
  // A zero value shall not be written (newDateTime may be zero if EXIF data is missing).
  if(newDateTime=0) then exit;
  // Frac returns time only, Trunc returns date only
  if Config.OptTime then dt := Frac(newDateTime) else dt := Frac(oldDateTime);
  if Config.OptDate then dt := dt + Trunc(newDateTime) else dt := dt + Trunc(oldDateTime);
  oldDateTime := dt;
end;

{*
 * Get the timestamp which is selected in Options => Source.
 * (Maybe multiple timestamps if "each selected target date" is chosen.)
 * The filesystem timestamp (created, modified, access) will always be retrieved.
 * @param filename The file or directory path.
 * @param timestamp Will contain the timestamps thereafter.
 * @param isDirectory true if directory, false if file
 * @param errs Returned errors.
 * @return true if at least created/modified/access timestamp were read successfully, false otherwise.
 }
function GetMyTimestamp(filename: String; var timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;
var
  ImgData: TImgData;
  Attrs: Integer;
  exif, exif_subresult: Boolean;
begin
  InitMyErrors(Config.curProcFileErrors);
  Config.curProcFile := filename;
  //errs.files := errs.files + 1;
  //if not FileExists(filename) then exit;

  InitTimestamp(timestamp);

  // FsTimestamp will always be retrieved!
  // (This is needed to preserve the FsTimestamp when only the file contents
  // should be changed, e.g. the EXIF timestamp.)
  Result := GetFsTimestamp(filename, timestamp, isDirectory);
  if not Result then begin
    // Access error (maybe file in use?)
    if isDirectory then begin
      errs.dirAccess := errs.dirAccess + 1;
      Config.curProcFileErrors.dirAccess := 1;
      if Config.OptErrDetails then Form4.Memo1.Lines.Add('ERR:DIRACCESS ' + filename);
    end else begin
      errs.fileAccess := errs.fileAccess + 1;
      Config.curProcFileErrors.fileAccess := 1;
      if Config.OptErrDetails then Form4.Memo1.Lines.Add('ERR:FILEACCESS ' + filename);
    end;
    exit;
  end;

  if not isDirectory then begin // isFile

    // In order to allow only date xor time being changed,
    // we must read EXIF data even if it should only be handled as target:
    if (Config.OptSource_EXIF_DateTime or Config.OptSource_EXIF_DateTimeOriginal or Config.OptSource_EXIF_DateTimeDigitized)
     or (Config.OptTarget_EXIF_DateTime or Config.OptTarget_EXIF_DateTimeOriginal or Config.OptTarget_EXIF_DateTimeDigitized)
     then begin
      exif := false;
      try
        ImgData := TimgData.Create();
        ImgData.BuildList := dEXIF.GenAll;

        if ImgData.ProcessFile(filename) then begin
          if ImgData.HasEXIF() then begin
            exif_subresult := true;
            if (Config.OptSource_EXIF_DateTime or Config.OptTarget_EXIF_DateTime) then begin
              timestamp.exifDT := ImgData.ExifObj.GetImgDT();
              if timestamp.exifDT = 0 then exif_subresult := false;
            end;
            if (Config.OptSource_EXIF_DateTimeOriginal or Config.OptTarget_EXIF_DateTimeOriginal) then begin
              timestamp.exifDTO := ImgData.ExifObj.GetImgDTO();
              if timestamp.exifDTO = 0 then exif_subresult := false;
            end;
            if (Config.OptSource_EXIF_DateTimeDigitized or Config.OptTarget_EXIF_DateTimeDigitized) then begin
              timestamp.exifDTD := ImgData.ExifObj.GetImgDTD();
              if timestamp.exifDTD = 0 then exif_subresult := false;
            end;
            // if a sub-result was false, the overall result stays false (so we can count an EXIF error below)
            if exif_subresult then exif := true;
          end;
        end;
      finally
        if Unprotect(filename, Attrs) then begin
          // Touching the file contents seems to change the last access date.
          // Thus, we need to reset it:
          SetFsTimestamp(filename, timestamp, isDirectory, false);
          Protect(filename, Attrs);
          // However, we cannot prohibit the Windows Explorer from
          // accessing/reloading a file whenever it was changed.
        end else begin
          errs.wprotected := errs.wprotected + 1;
          Config.curProcFileErrors.wprotected := 1;
          if Config.OptErrDetails then Form4.Memo1.Lines.Add('ERR:WRITEPROTECTED ' + filename);
        end;
        ImgData.Free;
        if not exif then begin
          errs.exifRead := errs.exifRead + 1;
          Config.curProcFileErrors.exifRead := 1;
          if Config.OptErrDetails then Form4.Memo1.Lines.Add('ERR:EXIFREAD ' + filename);
        end;
      end;
    end;

  end;

end;

{*
 * Set the timestamp/s which is/are selected in Options => Target.
 * @param filename The file or directory path.
 * @param myTimestamp The new timestamp/s.
 * @param isDirectory true if directory, false if file
 * @param errs Returned errors.
 * @return true if successful, false otherwise.
 }
function SetMyTimestamp(filename: String; timestamp: MyTimestamp; isDirectory: Boolean; var errs: MyErrors): Boolean;
var
  ImgData: TImgData;
  fileTimestamp: MyTimestamp;
  errs2: MyErrors;
  exif_result, exif_subresult: Boolean;
begin
  InitMyErrors(Config.curProcFileErrors);
  Config.curProcFile := filename;

  //errs.files := errs.files + 1;
  //if not FileExists(filename) then exit;

  // Remember: FsTimestamp MUST be read, as file may be edited for EXIF data and
  // may loose its original timestamp otherwise!

  // fetch old timestamp/s for adjustment
  InitMyErrors(errs2);
  Result := GetMyTimestamp(filename, fileTimestamp, isDirectory, errs2);
  if not Result then begin
    AddMyErrors(errs, errs2);
    exit;
  end;

  // Remember date (if undo is enabled)
  if Config.OptUndo then AddUndo(filename, fileTimestamp, isDirectory);

  // these three values should be set at once
  if Config.OptTarget_Created then UpdateDT(fileTimestamp.created, timestamp.created);
  if Config.OptTarget_Modified then UpdateDT(fileTimestamp.modified, timestamp.modified);
  if Config.OptTarget_Access then UpdateDT(fileTimestamp.access, timestamp.access);

  // Please note: UpdateDT is used here to modify the current target datetime
  // in order to preserve the target's original time or date
  // (if the user wants to change only the time OR only the date).

  // Special option: Directories: Additionally set created date
  // (if I recall correctly, Windows Explorer used to show created date for dirs)
  // Note: Added 'isDirectory and' here in v1.7 (was missing before)
  if isDirectory and Config.OptSetDirCreateTime and Config.OptTarget_Modified
   and (not Config.OptTarget_Created) then begin
    UpdateDT(fileTimestamp.created, timestamp.modified);
  end;

  // TODO maybe add support for IPTC tag

  // EXIF date may only be available for files, not for directories ;)
  if not isDirectory then begin // isFile
    if (Config.OptTarget_EXIF_DateTime or Config.OptTarget_EXIF_DateTimeOriginal or Config.OptTarget_EXIF_DateTimeDigitized) then begin
      exif_result := false;
      try
        ImgData := TimgData.Create();
        ImgData.BuildList := dEXIF.GenAll;

        if ImgData.ProcessFile(filename) then begin
          // Note: We can only overwrite existing EXIF tags, not add new ones!
          if ImgData.HasEXIF() then begin
            exif_subresult := true;
            if (Config.OptTarget_EXIF_DateTime) then begin
              UpdateDT(fileTimestamp.exifDT, timestamp.exifDT);
              if fileTimestamp.exifDT <> 0 then ImgData.ExifObj.OverwriteDT(fileTimestamp.exifDT)
              else exif_subresult := false;
            end;
            if (Config.OptTarget_EXIF_DateTimeOriginal) then begin
              UpdateDT(fileTimestamp.exifDTO, timestamp.exifDTO);
              if fileTimestamp.exifDTO <> 0 then ImgData.ExifObj.OverwriteDTO(fileTimestamp.exifDTO)
              else exif_subresult := false;
            end;
            if (Config.OptTarget_EXIF_DateTimeDigitized) then begin
              UpdateDT(fileTimestamp.exifDTD, timestamp.exifDTD);
              if fileTimestamp.exifDTD <> 0 then ImgData.ExifObj.OverwriteDTD(fileTimestamp.exifDTD)
              else exif_subresult := false;
            end;
            ImgData.WriteEXIFJpeg(filename);
            // if a sub-result was false, the overall result stays false (so we can count an EXIF error below)
            if exif_subresult = true then exif_result := true;
          end;
        end;
      finally
        ImgData.Free;
        if not exif_result then begin
          errs.exifWrite := errs.exifWrite + 1;
          Config.curProcFileErrors.exifWrite := 1;
          if Config.OptErrDetails then Form4.Memo1.Lines.Add('ERR:EXIFWRITE ' + filename);
        end;
      end;

    end;

  end;

  // This MUST be the last operation, as any editing may destroy
  // the FsTimestamp!
  Result := SetFsTimestamp(filename, fileTimestamp, isDirectory, false) and Result;

end;

{*
 * called by OpenFileDateNow (single file only):
 * Get date and time for file.
 * @param FileName The file path.
 * @param DateTime The file's timestamp.
 * @param init Initial call (true) or not (false).
 * @return true if successful, false otherwise.
 }
function getFileDateTime(FileName: String; var DateTime: TDateTime; init: Boolean): Boolean;
var
  timestamp: MyTimestamp;
begin
  if (init) then InitCurProcErrors();
  Result := GetMyTimestamp(filename, timestamp, false, Config.curProcErrors);
  // Exif errors must be detected separately:
  if ((Config.curProcErrors.exifRead > 0) and
   (Config.OptSource_EXIF_DateTime or
    Config.OptSource_EXIF_DateTimeOriginal or
    Config.OptSource_EXIF_DateTimeDigitized) ) then
   Result := false;
  if Result then begin
    if Config.OptSource_Created then DateTime := timestamp.created
    else if Config.OptSource_Modified then DateTime := timestamp.modified
    else if Config.OptSource_Access then DateTime := timestamp.access
    else if Config.OptSource_EXIF_DateTime then DateTime := timestamp.exifDT
    else if Config.OptSource_EXIF_DateTimeOriginal then DateTime := timestamp.exifDTO
    else if Config.OptSource_EXIF_DateTimeDigitized then DateTime := timestamp.exifDTD
    else Result := false;
  end;
end;

{*
 * called by OpenDirDateNow (single file only):
 * Get date and time for directory.
 * @param filename The directory path.
 * @param DateTime The file's timestamp.
 * @param init Initial call (true) or not (false).
 * @return true if successful, false otherwise.
 }
function getDirDateTime(filename: String; var DateTime: TDateTime; init: Boolean): Boolean;
var
  timestamp: MyTimestamp;
begin
  if (init) then InitCurProcErrors();
  Result := GetMyTimestamp(filename, timestamp, true, Config.curProcErrors);
  if Result then begin
    if Config.OptSource_Created then DateTime := timestamp.created
    else if Config.OptSource_Modified then DateTime := timestamp.modified
    else if Config.OptSource_Access then DateTime := timestamp.access
    //else if Config.OptSource_EXIF then //ignore
    else Result := false;
  end;
end;

{*
 * called by SaveDateNow (single file only):
 * save datetime to file
 * @param filename The file path.
 * @param DateTime The file's new timestamp.
 * @param init Initial call (true) or not (false).
 * @return true if successful, false otherwise.
 }
function setFileDateTime(filename: String; DateTime: TDateTime; init: Boolean): Boolean;
var
  timestamp: MyTimestamp;
begin
  if (init) then InitCurProcErrors();
  if (init) and Config.OptUndo then InitUndo();
  if Config.OptTarget_Created then timestamp.created := DateTime;
  if Config.OptTarget_Modified then timestamp.modified := DateTime;
  if Config.OptTarget_Access then timestamp.access := DateTime;
  if Config.OptTarget_EXIF_DateTime then timestamp.exifDT := DateTime;
  if Config.OptTarget_EXIF_DateTimeOriginal then timestamp.exifDTO := DateTime;
  if Config.OptTarget_EXIF_DateTimeDigitized then timestamp.exifDTD := DateTime;
  Result := SetMyTimestamp(filename, timestamp, false, Config.curProcErrors);
end;

{*
 * save datetime to directory
 * @param filename The directory path.
 * @param DateTime The directory's new timestamp.
 * @param init Initial call (true) or not (false).
 * @return true if successful, false otherwise.
 }
function setDirDateTime(filename: String; DateTime: TDateTime; init: Boolean): Boolean;
var
  timestamp: MyTimestamp;
begin
  if (init) then InitCurProcErrors();
  if (init) and Config.OptUndo then InitUndo();
  Result := true;
  if Config.OptTarget_Created then timestamp.created := DateTime
  else if Config.OptTarget_Modified then timestamp.modified := DateTime
  else if Config.OptTarget_Access then timestamp.access := DateTime
  else Result := false;
  if not Result then exit;
  Result := SetMyTimestamp(filename, timestamp, true, Config.curProcErrors);
end;

{*
 * Increase the values of a TDateTime object.
 * Each part of the timestamp (second, minute, ...) will be increased
 * individually.
 * @param DateTime The timestamp to be increased.
 * @param inc The increase time.
 * @return The increased timestamp.
 }
function IncDateTime(DateTime: TDateTime; inc: MyIncTime): TDateTime;
begin
  DateTime := IncSecond(DateTime,inc.s);
  DateTime := IncMinute(DateTime,inc.m);
  DateTime := IncHour(DateTime,inc.h);
  DateTime := IncDay(DateTime,inc.d);
  DateTime := IncYear(DateTime,inc.a);
  Result := DateTime;
end;

{*
 * Set dates for multiple files.
 * @param list A list of files.
 * @param inc The increase time.
 * @param DateTime The new timestamp.
 * @return The error count.
 }
function SetList(list: TStrings; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
var
  count:integer;
begin
  InitCurProcErrors();
  if Config.OptUndo then InitUndo();

  For count := 0 to list.Count-1 do begin
    Application.ProcessMessages; if(Config.curProcCancelled) then break;
    //process file
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(list[count]);
    Config.curProcErrors.files := Config.curProcErrors.files + 1;
    SetMyTimestampToolOpts(list[count], inc, DateTime, false);
  end;
  Result := Config.curProcErrors;
end;

{*
 * Set dates recursively (breadth first).
 * Not implemented recursively, but using a queue.
 * @param directory The directory path.
 * @param inc The increase time.
 * @param DateTime The new timestamp.
 * @return The error count.
 }
function SetRecursiveBF(directory: String; inc: MyIncTime; var DateTime: TDateTime): MyErrors;
var
  filenameptr, filenameptr2: ^String;
  SearchRec: TSearchRec;
  WorkList: TQueue;
  DirList: TStringList;
  FileList: TStringList;
  count: Integer;
  fa: Integer;
label
  AddDirList, AddFileList, EndAddLists;
begin
  InitCurProcErrors();
  if Config.OptUndo then InitUndo();

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  New(filenameptr);
  filenameptr^ := directory;

  WorkList := TQueue.Create;
  WorkList.Push(filenameptr);

  while WorkList.Count>0  do begin
    Application.ProcessMessages;
    filenameptr := WorkList.Pop;
    if (Config.curProcCancelled) then begin
      // do nothing (WorkList cleanup will happen below)
    end else if(filenameptr^[length(filenameptr^)] = '\') then begin

      if Config.OptToolsSetDirTime then begin
        //process directory
        if Config.OptLogDetails then Form4.Memo1.Lines.Add(filenameptr^);
        Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
        SetMyTimestampToolOpts(filenameptr^, inc, DateTime, true);
      end;

      //recurse subdirs (breadth first)

      DirList := TStringList.Create;
      FileList := TStringList.Create;

      fa := faAnyFile;
      if not Config.OptHidden then fa := fa xor faHidden;
      if not Config.OptSystem then fa := fa xor faSysFile;

      if FindFirst(filenameptr^ + '*.*', fa, SearchRec) = 0 then begin
        repeat
          Application.ProcessMessages; if(Config.curProcCancelled) then break;
          if (SearchRec.Attr and faDirectory = faDirectory)
          and (not (SearchRec.Name[1] = '.')) then begin
            //Current entry is a directory
            DirList.Add(filenameptr^+SearchRec.Name);
          end else begin
            //entry is a file
            if not (SearchRec.Name[1] = '.') then begin
              FileList.Add(filenameptr^+SearchRec.Name);
            end;
          end;
        until (not (FindNext(SearchRec) = 0));
        FindClose(SearchRec);

        // A little "goto" programming for alternate order.
        if Config.OptFilesFirst then goto AddFileList;

        AddDirList:
        DirList.Sort;
        for count := 0 to DirList.Count-1 do begin
          Application.ProcessMessages; if(Config.curProcCancelled) then break;
          // We are not yet going recursive here... (i.e. breadth first)
          New(filenameptr2);
          filenameptr2^ := DirList.Strings[count];
          if not (filenameptr2^[length(filenameptr2^)] = '\') then
            filenameptr2^ := filenameptr2^ + '\';
          WorkList.Push(filenameptr2);
        end;
        DirList.Free;
        if Config.OptFilesFirst then goto EndAddLists;

        AddFileList:
        FileList.Sort;
        for count := 0 to FileList.Count-1 do begin
          Application.ProcessMessages; if(Config.curProcCancelled) then break;
          // Also no direct processing here, but adding file to the queue
          New(filenameptr2);
          filenameptr2^ := FileList.Strings[count];
          WorkList.Push(filenameptr2);
        end;
        FileList.Free;
        if Config.OptFilesFirst then goto AddDirList;

        EndAddLists:

      end;

    end else begin
      //process file
      if Config.OptLogDetails then Form4.Memo1.Lines.Add(filenameptr^);
      Config.curProcErrors.files := Config.curProcErrors.files + 1;
      SetMyTimestampToolOpts(filenameptr^, inc, DateTime, false);
    end;
    filenameptr^ := ''; // Strings must be cleared first!
    Dispose(filenameptr);
  end;

  WorkList.Free;

  Result := Config.curProcErrors;

end;

{*
 * Set dates recursively (depth first).
 * @param directory The directory path.
 * @param inc The increase time.
 * @param DateTime The new timestamp.
 * @param init Initial call (true) or not (false).
 * @return The error count.
 }
function SetRecursive(directory: String;
  inc: MyIncTime;
  var DateTime: TDateTime;
  init: Boolean): MyErrors;
var
  SearchRec: TSearchRec;
  DirList: TStringList;
  FileList: TStringList;
  count: Integer;
  fa: Integer;
label
  AddDirList, AddFileList, EndAddLists;
begin
  if (init) then InitCurProcErrors();
  if (init) and Config.OptUndo then InitUndo();

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  if Config.OptToolsSetDirTime then begin
    // process directory
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
    Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
    SetMyTimestampToolOpts(directory, inc, DateTime, true);
  end;

  //recurse subdirs (depth first)

  DirList := TStringList.Create;
  FileList := TStringList.Create;

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        DirList.Add(directory+SearchRec.Name);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          FileList.Add(directory+SearchRec.Name);
        end;
      end;
    until (not (FindNext(SearchRec) = 0));
    FindClose(SearchRec);

    // A little "goto" programming for alternate order.
    if Config.OptFilesFirst then goto AddFileList;

    AddDirList:
    DirList.Sort;
    for count := 0 to DirList.Count-1 do begin
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      // process directory recursively (i.e. depth first)
      SetRecursive(DirList.Strings[count],inc,DateTime,false);
    end;
    DirList.Free;
    if Config.OptFilesFirst then goto EndAddLists;

    AddFileList:
    FileList.Sort;
    for count := 0 to FileList.Count-1 do begin
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      // process file
      if Config.OptLogDetails then Form4.Memo1.Lines.Add(FileList.Strings[count]);
      Config.curProcErrors.files := Config.curProcErrors.files + 1;
      SetMyTimestampToolOpts(FileList.Strings[count], inc, DateTime, false);
    end;
    FileList.Free;
    if Config.OptFilesFirst then goto AddDirList;

    EndAddLists:

  end;

  Result := Config.curProcErrors;

end;

{*
 * Copy directory dates from one directory to another (recursively).
 * @param directory A subdir of the source directory to be copied.
 * @param directory1 The source directory path.
 * @param directory2 The target directory path.
 * @param init Initial call (true) or not (false).
 * @return The error count.
 }
function CopyRecursive(directory, directory1, directory2: String; init: Boolean): MyErrors;
var
  SearchRec: TSearchRec;
  relpath: String;
  count: Integer;
  timestamp: MyTimestamp;
  fa: Integer;
begin
  if (init) then InitCurProcErrors();
  if (init) and Config.OptUndo then InitUndo();

  if not (directory[length(directory)] = '\') then
    directory := directory + '\';

  if not (directory1[length(directory1)] = '\') then
    directory1 := directory1 + '\';

  if not (directory2[length(directory2)] = '\') then
    directory2 := directory2 + '\';

  if Config.OptToolsSetDirTime then begin
    // process directory
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory1);
    Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
    if GetMyTimestamp(directory, timestamp, true, curProcErrors) then
      relpath := Copy(directory, length(directory1) + 1, length(directory) - length(directory1) );
      SetMyTimestamp(directory2 + relpath, timestamp, true, curProcErrors)
  end;

  // recurse subdirs (depth first)

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        // process directory recursively (i.e. depth first)
        CopyRecursive(directory+SearchRec.Name, directory1, directory2,false);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory+SearchRec.Name);
          Config.curProcErrors.files := Config.curProcErrors.files + 1;
          if GetMyTimestamp(directory+SearchRec.Name, timestamp, false, curProcErrors) then begin
            relpath := directory+SearchRec.Name;
            relpath := Copy(relpath, length(directory1) + 1, length(relpath) - length(directory1) );
            SetMyTimestamp(directory2 + relpath, timestamp, false, curProcErrors)
          end;
        end;
      end;
    until (not (FindNext(SearchRec) = 0));
    FindClose(SearchRec);

  end;

  Result := Config.curProcErrors;

end;

{*
 * Copy directory dates from one directory to another (by filename/size).
 * @param directory1 The source directory path.
 * @param directory2 The target directory path.
 * @return The error count.
 }
function CopyNameSize(directory1, directory2: String): MyErrors;
var
  cancelled: Boolean;
begin
  InitCurProcErrors();
  if Config.OptUndo then InitUndo();

  // TODO Do we need this here? Copied from RememberRecursive(init).
  if not(Config.OptSource_EachTarget or Config.OptTarget_Source) then begin
    Config.OptTarget_Source := true;
    Config.OptTarget_Created := Config.OptSource_Created;
    Config.OptTarget_Modified := Config.OptSource_Modified;
    Config.OptTarget_Access := Config.OptSource_Access;
    Config.OptTarget_EXIF_DateTime := Config.OptSource_EXIF_DateTime;
    Config.OptTarget_EXIF_DateTimeOriginal := Config.OptSource_EXIF_DateTimeOriginal;
    Config.OptTarget_EXIF_DateTimeDigitized := Config.OptSource_EXIF_DateTimeDigitized;
  end;

  // Read timestamps
  Form4.Memo1.Lines.Add(Config.MsgReadSourceFiles);
  InitHashList();
  CopyNameSizeReadRecursive(directory1);

  // Store timestamps
  if Config.curProcCancelled then begin
    InitCurProcErrors(); // also resets Config.curProcCancelled
  end else begin
    InitCurProcErrors();
    Form4.Memo1.Lines.Add(Config.MsgWriteTargetFiles);
    CopyNameSizeWriteRecursive(directory2);
  end;

  // Free memory and return.
  DestroyHashList();
  Result := Config.curProcErrors;

end;

{*
 * Store dates in list object.
 * You MUST call InitHashList first!
 * @param directory The directory path.
 }
function CopyNameSizeReadRecursive(directory: String): MyErrors;
var
  SearchRec: TSearchRec;
  datefileptr: ^MyDateFile;
  timestamp: MyTimestamp;
  fa: Integer;
  hashkey: String;
begin

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  try
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
    Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
  finally
  end;

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        CopyNameSizeReadRecursive(directory+SearchRec.Name);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory+SearchRec.Name);
          Config.curProcErrors.files := Config.curProcErrors.files + 1;
          // Remember file date (by filename and size).
          hashkey := SearchRec.Name + ':' + IntToStr(SearchRec.Size);
          if HashList.IndexOf(hashkey) = -1 then begin
            if GetMyTimestamp(directory+SearchRec.Name, timestamp, false, Config.curProcErrors) then begin
              New(datefileptr);
              datefileptr^.timestamp := timestamp;
              datefileptr^.filename := directory+SearchRec.Name;
              datefileptr^.isDirectory := false;
              HashList.AddObject(hashkey, TObject(datefileptr));
            end;
          end;
        end;
      end;
    until ( not (FindNext(SearchRec) = 0) );
    FindClose(SearchRec);
  end;

  Result := Config.curProcErrors;
end;

{*
 * Save dates from list object.
 * You MUST call CopyNameSizeReadRecursive first!
 * @param directory The directory path.
 }
function CopyNameSizeWriteRecursive(directory: String): MyErrors;
var
  SearchRec: TSearchRec;
  datefileptr: ^MyDateFile;
  timestamp: MyTimestamp;
  fa: Integer;
  hashkey: String;
begin

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  try
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
    Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
  finally
  end;

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        CopyNameSizeWriteRecursive(directory+SearchRec.Name);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory+SearchRec.Name);
          Config.curProcErrors.files := Config.curProcErrors.files + 1;
          // Get file date (by filename and size).
          hashkey := SearchRec.Name + ':' + IntToStr(SearchRec.Size);
          if HashList.IndexOf(hashkey) = -1 then begin
            Config.curProcErrors.fileNoSource := Config.curProcErrors.fileNoSource + 1;
            if Config.OptErrDetails then Form4.Memo1.Lines.Add('ERR:NOSOURCE ' + directory+SearchRec.Name);
          end else begin
            datefileptr := Pointer(HashList.Objects[HashList.IndexOf(hashkey)]);
            SetMyTimestamp(directory+SearchRec.Name, datefileptr^.timestamp, false, Config.curProcErrors);
          end;
        end;
      end;
    until ( not (FindNext(SearchRec) = 0) );
    FindClose(SearchRec);
  end;

  Result := Config.curProcErrors;
end;

{*
 * Set directory dates, each according to its newest or oldest file.
 * @param directory The directory path.
 * @param extremum Current extremum date.
 * @param newest Set to newest (true) or oldest (false) date.
 * @param init Initial call (true) or not (false).
 * @return The error count.
 }
function SetDirRecursiveLatest(directory: String; MainDateTime: TDateTime; var extremum: TDateTime; newest: Boolean; init: Boolean): MyErrors;
var
  SearchRec: TSearchRec;
  DirList: TStringList;
  FileList: TStringList;
  count: Integer;
  fa: Integer;
  DateTime: TDateTime;
  //latest: TDateTime;
  extremumSubdir: TDateTime;
  extremumUndefined: Boolean;
begin
  if (init) then InitCurProcErrors();
  if (init) and Config.OptUndo then InitUndo();

  extremumUndefined := true;

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  //recurse subdirs (depth first)

  DirList := TStringList.Create;
  FileList := TStringList.Create;

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
  Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        DirList.Add(directory+SearchRec.Name);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          FileList.Add(directory+SearchRec.Name);
        end;
      end;
    until (not (FindNext(SearchRec) = 0));
    FindClose(SearchRec);

    DirList.Sort;
    for count := 0 to DirList.Count-1 do begin
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      // process directory recursively (i.e. depth first)
      SetDirRecursiveLatest(DirList.Strings[count], MainDateTime, extremumSubdir, newest, false);
      if (Config.OptLatestSubdir) and (
        extremumUndefined
        or (newest and (extremumSubdir > extremum))
        or ((not newest) and (extremumSubdir < extremum))
      ) then begin
        extremum := extremumSubdir;
        extremumUndefined := false;
      end;
    end;
    DirList.Free;

    FileList.Sort;
    for count := 0 to FileList.Count-1 do begin
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      // process file - no changes here, thus we do not count files
      //Config.curProcErrors.files := Config.curProcErrors.files + 1;
      getFileDateTime(FileList.Strings[count], DateTime, false);
      if (
        extremumUndefined
        or (newest and (DateTime > extremum))
        or ((not newest) and (DateTime < extremum))
      ) then begin
        extremum := DateTime;
        extremumUndefined := false;
      end;
    end;
    FileList.Free;

  end;

  if (Config.OptMainTime2) and (extremum <= 0) then begin
    extremum := MainDateTime;
  end;

  if (extremum <= 0) then begin
      Config.curProcErrors.dirAccess := Config.curProcErrors.dirAccess + 1;
  end else begin
    if (not setDirDateTime(directory, extremum, false)) then begin
      Config.curProcErrors.dirAccess := Config.curProcErrors.dirAccess + 1;
    end;
  end;

  Result := Config.curProcErrors;

end;

{*
 * Init list for remembering dates.
 }
procedure InitDatesList();
begin
  //if DatesList<>nil then try
  //  DatesList.Free;
  //finally
  //end;
  DestroyDatesList();
  DatesList := TList.Create;
  // Also remember which dates have been remembered:
  Config.DatesListSettings.created := Config.OptTarget_Created;
  Config.DatesListSettings.modified := Config.OptTarget_Modified;
  Config.DatesListSettings.access := Config.OptTarget_Access;
  Config.DatesListSettings.exifDT := Config.OptTarget_EXIF_DateTime;
  Config.DatesListSettings.exifDTO := Config.OptTarget_EXIF_DateTimeOriginal;
  Config.DatesListSettings.exifDTD := Config.OptTarget_EXIF_DateTimeDigitized;
end;

{*
 * Destroy list for remembering dates.
 }
procedure DestroyDatesList();
var
  count: Integer;
  datefileptr: ^MyDateFile;
begin
  if DatesList<>nil then begin
    try
      for count := 0 to DatesList.Count - 1 do begin
        datefileptr := DatesList.Items[count];
        datefileptr^.filename := ''; // Strings must be cleared first!
        Dispose(datefileptr);
      end;
    finally
      DatesList.Free;
    end;
  end;
end;

{*
 * Init hash list for remembering dates.
 }
procedure InitHashList();
begin
  //DestroyHashList();
  HashList := THashedStringList.Create;
end;

{*
 * Destroy hash list for remembering dates.
 }
procedure DestroyHashList();
var
  count: Integer;
  datefileptr: ^MyDateFile;
begin
  if HashList<>nil then begin
    try
    // TODO dispose objects
      for count := 0 to HashList.Count - 1 do begin
        datefileptr := Pointer(HashList.Objects[count]);
        datefileptr^.filename := ''; // Strings must be cleared first!
        Dispose(datefileptr);
      end;
    finally
      HashList.Free;
    end;
  end;
end;

{*
 * Init undo buffer list on program start.
 }
procedure InitUndoList();
begin
  UndoList := TList.Create;
end;

{*
 * Destroy undo buffer list on program exit.
 }
procedure DestroyUndoList();
var
  count, count2: Integer;
  undoptr: ^MyUndo;
  datefileptr: ^MyDateFile;
begin
  if UndoList<>nil then begin
    try
      for count := 0 to UndoList.Count - 1 do begin
        undoptr := UndoList.Items[count];
        try
          for count2 := 0 to undoptr^.DatesList.Count - 1 do begin
            datefileptr := undoptr^.DatesList.Items[count2];
            datefileptr^.filename := ''; // Strings must be cleared first!
            Dispose(datefileptr);
          end;
        finally
          undoptr^.DatesList.Free;
          Dispose(undoptr);
        end;
      end;
    finally
      UndoList.Free;
    end;
  end;
end;

{*
 * Store dates in list object.
 * @param list A list of files.
 }
function RememberList(list: TStrings): MyErrors;
var count:integer;
  datefileptr: ^MyDateFile;
  timestamp: MyTimestamp;
begin
  if not(Config.OptSource_EachTarget or Config.OptTarget_Source) then begin
    Config.OptTarget_Source := true;
    Config.OptTarget_Created := Config.OptSource_Created;
    Config.OptTarget_Modified := Config.OptSource_Modified;
    Config.OptTarget_Access := Config.OptSource_Access;
    Config.OptTarget_EXIF_DateTime := Config.OptSource_EXIF_DateTime;
    Config.OptTarget_EXIF_DateTimeOriginal := Config.OptSource_EXIF_DateTimeOriginal;
    Config.OptTarget_EXIF_DateTimeDigitized := Config.OptSource_EXIF_DateTimeDigitized;
  end;

  InitCurProcErrors();
  InitDatesList();

  for count := 0 to list.Count-1 do begin
    Application.ProcessMessages; if(Config.curProcCancelled) then break;
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(list[count]);
    Config.curProcErrors.files := Config.curProcErrors.files + 1;
    if GetMyTimestamp(list[count], timestamp, false, Config.curProcErrors) then begin
      New(datefileptr);
      datefileptr^.timestamp := timestamp;
      datefileptr^.filename := list[count];
      datefileptr^.isDirectory := false;
      DatesList.Add(datefileptr);
    end;
  end;

  Result := Config.curProcErrors;
end;

{*
 * Store dates in list object.
 * You MUST call InitDatesList first!
 * @param directory The directory path.
 * @param init Initial call (true) or not (false).
 }
function RememberRecursive(directory: String; init: Boolean): MyErrors;
var
  SearchRec: TSearchRec;
  datefileptr: ^MyDateFile;
  timestamp: MyTimestamp;
  fa: Integer;
begin
  if (init) then begin
    if not(Config.OptSource_EachTarget or Config.OptTarget_Source) then begin
      Config.OptTarget_Source := true;
      Config.OptTarget_Created := Config.OptSource_Created;
      Config.OptTarget_Modified := Config.OptSource_Modified;
      Config.OptTarget_Access := Config.OptSource_Access;
      Config.OptTarget_EXIF_DateTime := Config.OptSource_EXIF_DateTime;
      Config.OptTarget_EXIF_DateTimeOriginal := Config.OptSource_EXIF_DateTimeOriginal;
      Config.OptTarget_EXIF_DateTimeDigitized := Config.OptSource_EXIF_DateTimeDigitized;
    end;
  end;

  if (init) then InitCurProcErrors();

  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  try
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
    Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
    // Remember directory date.
    if GetMyTimestamp(directory, timestamp, true, Config.curProcErrors) then begin
      New(datefileptr);
      datefileptr^.timestamp := timestamp;
      datefileptr^.filename := directory;
      datefileptr^.isDirectory := true;
      DatesList.Add(datefileptr);
    end;
  finally
  end;

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        RememberRecursive(directory+SearchRec.Name, false);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory+SearchRec.Name);
          Config.curProcErrors.files := Config.curProcErrors.files + 1;
          // Remember file date.
          if GetMyTimestamp(directory+SearchRec.Name, timestamp, false, Config.curProcErrors) then begin
            New(datefileptr);
            datefileptr^.timestamp := timestamp;
            datefileptr^.filename := directory+SearchRec.Name;
            datefileptr^.isDirectory := false;
            DatesList.Add(datefileptr);
          end;
        end;
      end;
    until ( not (FindNext(SearchRec) = 0) );
    FindClose(SearchRec);
  end;

  Result := Config.curProcErrors;
end;


{*
 * Save the remembered dates.
 * Same target settings will be used as when dates have been remembered.
 * @return errors.
 *}
function SaveRemembered(): MyErrors;
var count: Integer;
    Attrs: Integer;
    df: MyDateFile;
    info: String;
begin
  InitCurProcErrors();

  // Show to the user which settings will be used:
  info := '' + #13#10 + StrTargetDate + #13#10;
  if Config.DatesListSettings.created then info := info + StrOptCreated + #13#10;
  if Config.DatesListSettings.modified then info := info + StrOptModified + #13#10;
  if Config.DatesListSettings.access then info := info + StrOptLastAccess + #13#10;
  if Config.DatesListSettings.exifDT then info := info + StrOptExifDT + #13#10;
  if Config.DatesListSettings.exifDTO then info := info + StrOptExifDTO + #13#10;
  if Config.DatesListSettings.exifDTD then info := info + StrOptExifDTD + #13#10;

  if MessageDlg(MsgAdjustOptions + #13#10 + info, mtInformation, [mbOk, mbCancel], 0) = mrCancel then begin
    Result := Config.curProcErrors;
    Exit;
  end;

  Form4.Show();
  // LoadSettingsNow(''); // Reset settings to default - well, better not!

  // Always save date AND time
  Config.OptTime := true;
  Config.OptDate := true;

  Config.OptTarget_Created := Config.DatesListSettings.created;
  Config.OptTarget_Modified := Config.DatesListSettings.modified;
  Config.OptTarget_Access := Config.DatesListSettings.access;
  Config.OptTarget_EXIF_DateTime := Config.DatesListSettings.exifDT;
  Config.OptTarget_EXIF_DateTimeOriginal := Config.DatesListSettings.exifDTO;
  Config.OptTarget_EXIF_DateTimeDigitized := Config.DatesListSettings.exifDTD;

  // init undo AFTER changing settings:
  if Config.OptUndo then InitUndo();

  if (DatesList<>nil) and (DatesList.Count<>0) then begin
    For count := 0 to DatesList.Count-1 do begin
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      df := MyDateFile(DatesList.Items[count]^);
      if df.isDirectory then begin
        if Config.OptToolsSetDirTime then begin
          // set directory date
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(df.filename);
          Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
          SetMyTimestamp(df.filename, df.timestamp, df.isDirectory, Config.curProcErrors);
        end;
      end else begin
        // set file date
        if Config.OptLogDetails then Form4.Memo1.Lines.Add(df.filename);
        Config.curProcErrors.files := Config.curProcErrors.files + 1;
        if not Unprotect(df.filename, Attrs) then begin
          Config.curProcErrors.wprotected := Config.curProcErrors.wprotected + 1
        end else begin
            SetMyTimestamp(df.filename, df.timestamp, df.isDirectory, Config.curProcErrors);

            Protect(df.filename, Attrs);
        end;
      end;
    end;

  end;
  Result := Config.curProcErrors;
end;

{*
 * Forget the remembered dates.
 *}
procedure ForgetRemembered();
var
  count: Integer;
  datefileptr: ^MyDateFile;
begin
    if DatesList<>nil then begin
      try
        for count := 0 to DatesList.Count - 1 do begin
          datefileptr := DatesList.Items[count];
          datefileptr.filename := ''; // Strings must be cleared first!
          Dispose(datefileptr);
        end;
      finally
        DatesList.Clear;
      end;
    end;
end;

{*
 * Init MyTimestamp record with zeroes.
 * @param timestamp MyTimestamp record.
 *}
procedure InitTimestamp(var timestamp: MyTimestamp);
begin
  timestamp.created := 0;
  timestamp.modified := 0;
  timestamp.access := 0;
  timestamp.exifDT := 0;
  timestamp.exifDTO := 0;
  timestamp.exifDTD := 0;
end;

{*
 * Init MyErrors record with zeroes.
 * @param errs MyErrors record.
 *}
procedure InitMyErrors(var errs: MyErrors);
begin
  errs.files := 0;
  errs.dirs := 0;
  errs.wprotected := 0;
  errs.fileAccess := 0;
  errs.fileNoSource := 0;
  errs.dirAccess := 0;
  errs.exifRead := 0;
  errs.exifWrite := 0;
end;

{*
 * Init current process records.
 *}
procedure InitCurProcErrors();
begin
  InitMyErrors(Config.curProcErrors);
  InitMyErrors(Config.curProcFileErrors);
  Config.curProcCancelled := false;
  Config.curProcFile := '';
end;

{*
 * Add MyErrors record to another one.
 * @param target MyErrors target record.
 * @param source MyErrors source record.
 *}
procedure AddMyErrors(var target: MyErrors; source: MyErrors);
begin
  target.files := target.files + source.files;
  target.dirs := target.dirs + source.dirs;
  target.wprotected := target.wprotected + source.wprotected;
  target.fileAccess := target.fileAccess + source.fileAccess;
  target.fileNoSource := target.fileNoSource + source.fileNoSource;
  target.dirAccess := target.dirAccess + source.dirAccess;
  target.exifRead := target.exifRead + source.exifRead;
  target.exifWrite := target.exifWrite + source.exifWrite;
end;

{*
 * Get max values for two MyErrors records.
 }
function MaxMyErrors(errs1: MyErrors; errs2: MyErrors): MyErrors;
begin
  Result.files := Max(errs1.files, errs2.files);
  Result.dirs := Max(errs1.dirs, errs2.dirs);
  Result.wprotected := Max(errs1.wprotected, errs2.wprotected);
  Result.fileAccess := Max(errs1.fileAccess, errs2.fileAccess);
  Result.fileNoSource := Max(errs1.fileNoSource, errs2.fileNoSource);
  Result.dirAccess := Max(errs1.dirAccess, errs2.dirAccess);
  Result.exifRead := Max(errs1.exifRead, errs2.exifRead);
  Result.exifWrite := Max(errs1.exifWrite, errs2.exifWrite);
end;

{*
 * Sets timestamp with selected tool options (i.e. with 'increase time').
 * @param filename The file or directory path.
 * @param inc The increase time.
 * @param DateTime date and time to be set.
 * @param isDirectory true if directory, false if file
 * @param errs Returned errors.
 }
procedure SetMyTimestampToolOpts(filename: String; inc: MyIncTime; var DateTime: TDateTime; isDirectory: Boolean);
var
  timestamp: MyTimestamp;
  Attrs: Integer;
  myErrsGet: MyErrors;
  myErrsSet: MyErrors;
begin
  InitMyErrors(myErrsGet);
  InitMyErrors(myErrsSet);


  if isDirectory then begin

    // process directory

    //set directory date
    if GetMyTimestamp(filename, timestamp, true, myErrsGet) then begin
      if Config.OptOriginalTime = true then begin

        if Config.OptSource_EachTarget then begin

          if Config.OptTarget_Created
           then timestamp.created := IncDateTime(timestamp.created,inc);
          if Config.OptTarget_Modified
           then timestamp.modified := IncDateTime(timestamp.modified,inc);
          if Config.OptTarget_Access
           then timestamp.access := IncDateTime(timestamp.access,inc);

        end else begin
          if Config.OptSource_Created then DateTime := timestamp.created
          else if Config.OptSource_Modified then DateTime := timestamp.modified
          else if Config.OptSource_Access then DateTime := timestamp.access;
          DateTime := IncDateTime(DateTime,inc);

          if Config.OptTarget_Created then timestamp.created := DateTime;
          if Config.OptTarget_Modified then timestamp.modified := DateTime;
          if Config.OptTarget_Access then timestamp.access := DateTime;
        end;

        SetMyTimestamp(filename, timestamp, true, myErrsSet);
      end else begin
        if Config.OptTarget_Created then timestamp.created := DateTime;
        if Config.OptTarget_Modified then timestamp.modified := DateTime;
        if Config.OptTarget_Access then timestamp.access := DateTime;

        SetMyTimestamp(filename, timestamp, true, myErrsSet);

        DateTime := IncDateTime(DateTime,inc);
      end;

    end;


  end else begin

    // process file

    if not Unprotect(filename, Attrs) then begin
        Config.curProcErrors.wprotected := Config.curProcErrors.wprotected + 1;
    end else begin
      // Note: We do not cancel if one file has no EXIF data, as
      // GetMyTimestamp returns true if at least created/modified/access
      // was readable...
      if GetMyTimestamp(filename, timestamp, false, myErrsGet) then begin
        if Config.OptOriginalTime {OptOriginalTime.checked = true} then begin
          if Config.OptSource_EachTarget then begin

            if Config.OptTarget_Created
             then timestamp.created := IncDateTime(timestamp.created,inc);
            if Config.OptTarget_Modified
             then timestamp.modified := IncDateTime(timestamp.modified,inc);
            if Config.OptTarget_Access
             then timestamp.access := IncDateTime(timestamp.access,inc);
            if Config.OptTarget_EXIF_DateTime and (timestamp.exifDT <> 0)
             then timestamp.exifDT := IncDateTime(timestamp.exifDT,inc);
            if Config.OptTarget_EXIF_DateTimeOriginal and (timestamp.exifDTO <> 0)
             then timestamp.exifDTO := IncDateTime(timestamp.exifDTO,inc);
            if Config.OptTarget_EXIF_DateTimeDigitized and (timestamp.exifDTD <> 0)
             then timestamp.exifDTD := IncDateTime(timestamp.exifDTD,inc);

          end else begin
            if Config.OptSource_Created then DateTime := timestamp.created
            else if Config.OptSource_Modified then DateTime := timestamp.modified
            else if Config.OptSource_Access then DateTime := timestamp.access
            else if Config.OptSource_EXIF_DateTime then DateTime := timestamp.exifDT
            else if Config.OptSource_EXIF_DateTimeOriginal then DateTime := timestamp.exifDTO
            else if Config.OptSource_EXIF_DateTimeDigitized then DateTime := timestamp.exifDTD;
            if DateTime <> 0 then DateTime := IncDateTime(DateTime,inc);

            if Config.OptTarget_Created then timestamp.created := DateTime;
            if Config.OptTarget_Modified then timestamp.modified := DateTime;
            if Config.OptTarget_Access then timestamp.access := DateTime;
            if Config.OptTarget_EXIF_DateTime then timestamp.exifDT := DateTime;
            if Config.OptTarget_EXIF_DateTimeOriginal then timestamp.exifDTO := DateTime;
            if Config.OptTarget_EXIF_DateTimeDigitized then timestamp.exifDTD := DateTime;
          end;

          SetMyTimestamp(filename, timestamp, false, myErrsSet);
        end else begin
          if Config.OptTarget_Created then timestamp.created := DateTime;
          if Config.OptTarget_Modified then timestamp.modified := DateTime;
          if Config.OptTarget_Access then timestamp.access := DateTime;
          if Config.OptTarget_EXIF_DateTime then timestamp.exifDT := DateTime;
          if Config.OptTarget_EXIF_DateTimeOriginal then timestamp.exifDTO := DateTime;
          if Config.OptTarget_EXIF_DateTimeDigitized then timestamp.exifDTD := DateTime;

          SetMyTimestamp(filename, timestamp, false, myErrsSet);
          DateTime := IncDateTime(DateTime,inc);
        end;

        Protect(filename, Attrs);
      end;
    end;

  end;

  // Finally, sum up errors
  // (some errors may appear twice, once while getting and once while setting the timestamp)
  AddMyErrors(Config.curProcErrors, MaxMyErrors(myErrsGet,myErrsSet) );
end;

{*
 * Export remembered timestamps to CSV file.
 * @param filename The path to the CSV file to be exported.
 }
function ExportRememberedCsv(filename: String): MyErrors;
var f1: TextFile;
    line: String;
    count: Integer;
    df: MyDateFile;
begin
  InitCurProcErrors();
  try
    AssignFile(f1, filename);
    {$I-}
    Rewrite(f1);
    {$I+}
    if ioresult=0 then begin
      // Write header:
      line := '"file"';
      if Config.DatesListSettings.created then line := line + ',"created"';
      if Config.DatesListSettings.modified then line := line + ',"modified"';
      if Config.DatesListSettings.access then line := line + ',"lastaccess"';
      if Config.DatesListSettings.exifDT then line := line + ',"exifdt"';
      if Config.DatesListSettings.exifDTO then line := line + ',"exifdto"';
      if Config.DatesListSettings.exifDTD then line := line + ',"exifdtd"';
      WriteLn(f1, line);

      // Write data:

      if (DatesList<>nil) and (DatesList.Count<>0) then begin
        For count := 0 to DatesList.Count-1 do begin
          Application.ProcessMessages; if(Config.curProcCancelled) then break;
          df := MyDateFile(DatesList.Items[count]^);
          if df.isDirectory then begin
            //if Config.OptToolsSetDirTime then begin
              // save directory date
              if Config.OptLogDetails then Form4.Memo1.Lines.Add(df.filename);
              Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
              line := '"' + df.filename + '"';
              if Config.DatesListSettings.created then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.created) + '"';
              if Config.DatesListSettings.modified then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.modified) + '"';
              if Config.DatesListSettings.access then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.access) + '"';
              if Config.DatesListSettings.exifDT then line := line + ',""';
              if Config.DatesListSettings.exifDTO then line := line + ',""';
              if Config.DatesListSettings.exifDTD then line := line + ',""';
              WriteLn(f1, line);
            //end;
          end else begin
            // save file date
            if Config.OptLogDetails then Form4.Memo1.Lines.Add(df.filename);
            Config.curProcErrors.files := Config.curProcErrors.files + 1;
            line := '"' + df.filename + '"';
            if Config.DatesListSettings.created then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.created) + '"';
            if Config.DatesListSettings.modified then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.modified) + '"';
            if Config.DatesListSettings.access then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.access) + '"';
            if Config.DatesListSettings.exifDT then begin
              if df.timestamp.exifDT=0 then begin
                line := line + ',""';
              end else begin
                line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.exifDT) + '"';
              end;
            end;
            if Config.DatesListSettings.exifDTO then begin
              if df.timestamp.exifDTO=0 then begin
                line := line + ',""';
              end else begin
                line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.exifDTO) + '"';
              end;
            end;
            if Config.DatesListSettings.exifDTD then begin
              if df.timestamp.exifDTD=0 then begin
                line := line + ',""';
              end else begin
                line := line + ',"' + FormatDateTime('yyyymmddhhnnss', df.timestamp.exifDTD) + '"';
              end;
            end;
            WriteLn(f1, line);
          end;
        end;
      end;
    end else begin
      Form4.Memo1.Lines.Add(MsgCsvWriteErr);
    end;
  finally
    {$I-}
    closefile(f1);
    {$I+}
    if ioresult=0 then begin end; //Reset ioresult
  end;

  Result := Config.curProcErrors;
end;

{*
 * Save dates to CSV file.
 * @param directory The directory path.
 * @param filename The CSV file path.
 }
function SaveCsv(directory, filename: String): MyErrors;
var f1: TextFile;
    line: String;
begin
  InitCurProcErrors();
  try
    AssignFile(f1, filename);
    {$I-}
    Rewrite(f1);
    {$I+}
    if ioresult=0 then begin
      line := '"file"';
      if Config.OptTarget_Created then line := line + ',"created"';
      if Config.OptTarget_Modified then line := line + ',"modified"';
      if Config.OptTarget_Access then line := line + ',"lastaccess"';
      if Config.OptTarget_EXIF_DateTime then line := line + ',"exifdt"';
      if Config.OptTarget_EXIF_DateTimeOriginal then line := line + ',"exifdto"';
      if Config.OptTarget_EXIF_DateTimeDigitized then line := line + ',"exifdtd"';
      WriteLn(f1,line);
      SaveCsvRecursive(directory, f1);
    end else begin
      Form4.Memo1.Lines.Add(MsgCsvWriteErr);
    end;
  finally
    {$I-}
    closefile(f1);
    {$I+}
    if ioresult=0 then begin end; //Reset ioresult
  end;

  Result := Config.curProcErrors;
end;

{*
 * Save dates to CSV file - please call SaveCsv to run.
 * @param directory The directory path.
 * @param f1 The output file.
 }
procedure SaveCsvRecursive(directory: String; var f1: TextFile);
var
  SearchRec: TSearchRec;
  timestamp: MyTimestamp;
  fa: Integer;
  line: String;
begin
  if not (directory[length(directory)] = '\') then
  directory := directory + '\';

  try
    // Get directory date.
    if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
    Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
    if GetMyTimestamp(directory, timestamp, true, Config.curProcErrors) then begin
      line := '"' + directory + '"';
      if Config.OptTarget_Created then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.created) + '"';
      if Config.OptTarget_Modified then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.modified) + '"';
      if Config.OptTarget_Access then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.access) + '"';
      if Config.OptTarget_EXIF_DateTime then line := line + ',""';
      if Config.OptTarget_EXIF_DateTimeOriginal then line := line + ',""';
      if Config.OptTarget_EXIF_DateTimeDigitized then line := line + ',""';
      WriteLn(f1, line);
    end;
  finally
  end;

  fa := faAnyFile;
  if not Config.OptHidden then fa := fa xor faHidden;
  if not Config.OptSystem then fa := fa xor faSysFile;

  if FindFirst(directory + '*.*', fa, SearchRec) = 0 then begin
    repeat
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      if (SearchRec.Attr and faDirectory = faDirectory)
      and (not (SearchRec.Name[1] = '.')) then begin
        //Current entry is a directory
        SaveCsvRecursive(directory+SearchRec.Name, f1);
      end else begin
        //entry is a file
        if not (SearchRec.Name[1] = '.') then begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory+SearchRec.Name);
          Config.curProcErrors.files := Config.curProcErrors.files + 1;
          if GetMyTimestamp(directory+SearchRec.Name, timestamp, false, Config.curProcErrors) then begin
            line := '"' + directory+SearchRec.Name + '"';
            if Config.OptTarget_Created then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.created) + '"';
            if Config.OptTarget_Modified then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.modified) + '"';
            if Config.OptTarget_Access then line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.access) + '"';
            // exif=0 will write an empty cell
            if Config.OptTarget_EXIF_DateTime then begin
              if timestamp.exifDT = 0 then begin
                line := line + ',""';
              end else begin
                line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.exifDT) + '"';
              end;
            end;
            if Config.OptTarget_EXIF_DateTimeOriginal then begin
              if timestamp.exifDTO = 0 then begin
                line := line + ',""';
              end else begin
                line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.exifDTO) + '"';
              end;
            end;
            if Config.OptTarget_EXIF_DateTimeDigitized then begin
              if timestamp.exifDTD = 0 then begin
                line := line + ',""';
              end else begin
                line := line + ',"' + FormatDateTime('yyyymmddhhnnss', timestamp.exifDTD) + '"';
              end;
            end;
            WriteLn(f1, line);
          end;
        end;
      end;
    until ( not (FindNext(SearchRec) = 0) );
    FindClose(SearchRec);
  end;
end;

{*
 * Import dates from CSV file.
 }
function ImportCsv(filename: String): MyErrors;
var f1: TextFile;
    line, directory, info: String;
    ValuesList: TStringList;
    n: Integer;
    fFile,fCreated, fModified, fLastaccess, fExifdt, fExifdto, fExifdtd: integer;
    timestamp: MyTimestamp;
begin
  InitCurProcErrors();

  // TODO Should we walk through the file and check all values first?

  n := 0;

  try
    AssignFile(f1, filename);
    {$I-}
    reset(f1);
    {$I+}
    if ioresult=0 then begin
      while not eof(f1) do begin
        n := n + 1;
        readln(f1, line);
        ValuesList := TStringList.Create;
        ValuesList.Delimiter := ',';
        ValuesList.QuoteChar := '"';
        ValuesList.DelimitedText := line;

        if n=1 then begin
          fFile := ValuesList.IndexOf('file');
          fCreated := ValuesList.IndexOf('created');
          fModified := ValuesList.IndexOf('modified');
          fLastaccess := ValuesList.IndexOf('lastaccess');
          fExifdt := ValuesList.IndexOf('exifdt');
          fExifdto := ValuesList.IndexOf('exifdto');
          fExifdtd := ValuesList.IndexOf('exifdtd');

          if (fFile=-1) or ((fCreated=-1) and (fModified=-1) and (fLastaccess=-1) and (fExifdt=-1) and (fExifdto=-1) and (fExifdtd=-1)) then begin
            info := ErrCsvImportHeaders + #13#10 + #13#10;
            info := info + '- file ' + StrColMandatory + #13#10;
            info := info + '- created' + #13#10;
            info := info + '- modified' + #13#10;
            info := info + '- lastaccess' + #13#10;
            info := info + '- exifdt' + #13#10;
            info := info + '- exifdto' + #13#10;
            info := info + '- exifdtd' + #13#10;
            MessageDlg(info, mtError, [mbOK], 0);
            ValuesList.Free;
            break;
          end;

          // Show to the user which settings will be used:
          info := '' + #13#10 + StrTargetDate + #13#10;
          if fCreated >= 0 then info := info + StrOptCreated + #13#10;
          if fModified >= 0 then info := info + StrOptModified + #13#10;
          if fLastaccess >= 0 then info := info + StrOptLastAccess + #13#10;
          if fExifdt >= 0 then info := info + StrOptExifDT + #13#10;
          if fExifdto >= 0 then info := info + StrOptExifDTO + #13#10;
          if fExifdtd >= 0 then info := info + StrOptExifDTD + #13#10;

          if MessageDlg(MsgAdjustOptions + #13#10 + info, mtInformation, [mbOk, mbCancel], 0) = mrCancel then begin
            ValuesList.Free;
            break;
          end;

          Form4.Show();

          // Always save date AND time
          Config.OptTime := true;
          Config.OptDate := true;

          Config.OptTarget_Created := (fCreated >= 0);
          Config.OptTarget_Modified := (fModified >= 0);
          Config.OptTarget_Access := (fLastaccess >= 0);
          Config.OptTarget_EXIF_DateTime := (fExifdt >= 0);
          Config.OptTarget_EXIF_DateTimeOriginal := (fExifdto >= 0);
          Config.OptTarget_EXIF_DateTimeDigitized := (fExifdtd >= 0);

          // init undo AFTER changing settings:
          if Config.OptUndo then InitUndo();

          continue;
        end;

        Application.ProcessMessages; if(Config.curProcCancelled) then break;

        timestamp.created := 0;
        timestamp.modified := 0;
        timestamp.access := 0;
        timestamp.exifDT := 0;
        timestamp.exifDTO := 0;
        timestamp.exifDTD := 0;
        if fCreated >= 0 then timestamp.created := MyStringToDateTime(ValuesList.Strings[fCreated]);
        if fModified >= 0 then timestamp.modified := MyStringToDateTime(ValuesList.Strings[fModified]);
        if fLastaccess >= 0 then timestamp.access := MyStringToDateTime(ValuesList.Strings[fLastaccess]);
        // empty string will be treated as exif=0
        if fExifdt >= 0 then begin
          if ValuesList.Strings[fExifdt] = '' then begin
            timestamp.exifDT := 0;
          end else begin
            timestamp.exifDT := MyStringToDateTime(ValuesList.Strings[fExifdt]);
          end;
        end;
        if fExifdto >= 0 then begin
          if ValuesList.Strings[fExifdto] = '' then begin
            timestamp.exifDTO := 0;
          end else begin
            timestamp.exifDTO := MyStringToDateTime(ValuesList.Strings[fExifdto]);
          end;
        end;
        if fExifdtd >= 0 then begin
          if ValuesList.Strings[fExifdtd] = '' then begin
            timestamp.exifDTD := 0;
          end else begin
            timestamp.exifDTD := MyStringToDateTime(ValuesList.Strings[fExifdtd]);
          end;
        end;

        directory := ValuesList.Strings[fFile];
        if not (directory[length(directory)] = '\') then begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
          Config.curProcErrors.files := Config.curProcErrors.files + 1;
          SetMyTimestamp(directory, timestamp, False, Config.curProcErrors);
        end else begin
          if Config.OptLogDetails then Form4.Memo1.Lines.Add(directory);
          Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
          SetMyTimestamp(directory, timestamp, True, Config.curProcErrors);
        end;

        ValuesList.Free;
      end;
    end else begin
      Form4.Memo1.Lines.Add(MsgCsvReadErr);
    end;
  finally
    {$I-}
    closefile(f1);
    {$I+}
    if ioresult=0 then begin end; //Reset ioresult
  end;

  if n=0 then begin
    MessageDlg(ErrCsvFileEmpty, mtError, [mbOK], 0);
  end;

  Result := Config.curProcErrors;
end;

{*
 * Convert string vom CSV file to TDateTime.
 }
function MyStringToDateTime(dtstring: String): TDateTime;
var dt: TDateTime;
    yyyy,mm,dd,hh,nn,ss,code: Integer;
begin
  dt := 0;
  Val(copy(dtstring,1,4),yyyy,code);
  Val(copy(dtstring,5,2),mm,code);
  Val(copy(dtstring,7,2),dd,code);
  Val(copy(dtstring,9,2),hh,code);
  Val(copy(dtstring,11,2),nn,code);
  Val(copy(dtstring,13,2),ss,code);
  ReplaceDate(dt,EncodeDate(yyyy,mm,dd));
  ReplaceTime(dt,EncodeTime(hh,nn,ss,0));
  Result := dt;
end;

{*
 * Init "undo" buffer for current process.
 *
 }
procedure InitUndo();
var
  undoptr: ^MyUndo;
begin
  New(undoptr);
  // remember current time (for showing 'Undo changes from ddmmyyyy'):
  undoptr^.singleTimestamp := Now();
  // init a MyDateFile list:
  undoptr^.DatesList := TList.Create;
  // init a MyProcessed record:
  undoptr^.DatesListSettings.created := Config.OptTarget_Created;
  undoptr^.DatesListSettings.modified := Config.OptTarget_Modified;
  undoptr^.DatesListSettings.access := Config.OptTarget_Access;
  undoptr^.DatesListSettings.exifDT := Config.OptTarget_EXIF_DateTime;
  undoptr^.DatesListSettings.exifDTO := Config.OptTarget_EXIF_DateTimeOriginal;
  undoptr^.DatesListSettings.exifDTD := Config.OptTarget_EXIF_DateTimeDigitized;
  UndoList.Add(undoptr);
end;

{*
 * Add timestamp to "undo" buffer.
 * @param filename
 * @param timestamp
 }
procedure AddUndo(filename: String; timestamp: MyTimestamp; isDirectory: boolean);
var
  datefileptr: ^MyDateFile;
  undoptr: ^MyUndo;
begin
  // add timestamp to list
  New(datefileptr);
  datefileptr^.timestamp := timestamp;
  datefileptr^.filename := filename;
  datefileptr^.isDirectory := isDirectory;
  undoptr := UndoList.Last;
  undoptr^.DatesList.Add(datefileptr);
end;

{*
 * Undo changes.
 }
function Undo(): MyErrors;
var count: Integer;
  undoptr: ^MyUndo;
  datefileptr: ^MyDateFile;
  info: String;
  df: MyDateFile;
  Attrs: Integer;
  lastOptUndo: Boolean;
begin
  InitCurProcErrors();

  undoptr := UndoList.Last;

  info := ReplaceString(AskReallyUndo, '%n%', Stri(undoptr^.DatesList.Count));
  info := ReplaceString(info, '%t%', FormatDateTime('dd.mm.yyyy hh:nn:ss', undoptr^.singleTimestamp));
  info := info + #13#10#13#10 + StrTargetDate + #13#10;
  if undoptr^.DatesListSettings.created then info := info + StrOptCreated + #13#10;
  if undoptr^.DatesListSettings.modified then info := info + StrOptModified + #13#10;
  if undoptr^.DatesListSettings.access then info := info + StrOptLastAccess + #13#10;
  if undoptr^.DatesListSettings.exifDT then info := info + StrOptExifDT + #13#10;
  if undoptr^.DatesListSettings.exifDTO then info := info + StrOptExifDTO + #13#10;
  if undoptr^.DatesListSettings.exifDTD then info := info + StrOptExifDTD + #13#10;

  if MessageDlg(info, mtConfirmation, [mbOk, mbCancel], 0) = mrOk then begin


    // Always save date AND time
    Config.OptTime := true;
    Config.OptDate := true;

    // Use settings from undo buffer
    Config.OptTarget_Created := undoptr^.DatesListSettings.created;
    Config.OptTarget_Modified := undoptr^.DatesListSettings.modified;
    Config.OptTarget_Access := undoptr^.DatesListSettings.access;
    Config.OptTarget_EXIF_DateTime := undoptr^.DatesListSettings.exifDT;
    Config.OptTarget_EXIF_DateTimeOriginal := undoptr^.DatesListSettings.exifDTO;
    Config.OptTarget_EXIF_DateTimeDigitized := undoptr^.DatesListSettings.exifDTD;

    Form4.Show();

    // Preserve last OptUndo state, and disable OptUndo while performing undo.
    lastOptUndo := Config.OptUndo;
    Config.OptUndo := False;

    for count := 0 to undoptr^.DatesList.Count - 1 do begin
      Application.ProcessMessages; if(Config.curProcCancelled) then break;
      df := MyDateFile(undoptr^.DatesList.Items[count]^);
      if df.isDirectory then begin
        // set directory date
        if Config.OptLogDetails then Form4.Memo1.Lines.Add(df.filename);
        Config.curProcErrors.dirs := Config.curProcErrors.dirs + 1;
        SetMyTimestamp(df.filename, df.timestamp, df.isDirectory, Config.curProcErrors);
      end else begin
        // set file date
        if Config.OptLogDetails then Form4.Memo1.Lines.Add(df.filename);
        Config.curProcErrors.files := Config.curProcErrors.files + 1;
        if not Unprotect(df.filename, Attrs) then begin
          Config.curProcErrors.wprotected := Config.curProcErrors.wprotected + 1
        end else begin
            SetMyTimestamp(df.filename, df.timestamp, df.isDirectory, Config.curProcErrors);

            Protect(df.filename, Attrs);
        end;
      end;
    end;

    Config.OptUndo := lastOptUndo;

    if not(Config.curProcCancelled) then begin
      // Remove undo operation:
      if UndoList.Count > 0 then begin
        try
          for count := 0 to undoptr^.DatesList.Count - 1 do begin
            datefileptr := undoptr^.DatesList.Items[count];
            datefileptr^.filename := '';
            Dispose(datefileptr);
          end;
        finally
          undoptr^.DatesList.Free;
          UndoList.Delete(UndoList.Count - 1);
        end;
      end;
    end;

  end;

  Result := Config.curProcErrors;
end;

{*
 * Check if there are undo values available.
 }
function HasUndo(): Boolean;
begin
  Result := false;
  if UndoList.Count > 0 then begin
    Result := true;
  end;
end;


end.
