/**
 *  TETRIS clone coded by Manuel Haim, February 2003.
 *  Ported from Delphi(Win32) to Pascal(DOS),
 *  then ported from Pascal(DOS) to Java(platform-independent).
 *  v0.2 of March 2004, released as OPEN SOURCE software.
 *
 *  Source modified using BlueJ.
 *  Visit http://McSquirrel.istcool.de
 **/

// Importing needed classes:
import java.applet.Applet;
import java.awt.*;
import java.awt.event.*;

public class McTris
extends Applet // For this is an applet!!!
implements KeyListener, // For handling an object of this class
                  // as KeyListener to listen to keyboard input.
Runnable // For handling an object of this class as a Thread.
{
    /** Program's version number. */
    private final String VERSION = "0.2";
    
    /** The offscreen image for flicker-free painting. */
    private Image dbImage;
    /** The graphics belongs to the image. */
    private Graphics dbGraphics;
    
    /** The thread for moving the current stone downwards. */
    private Thread th;
    
    /** This matrix is the playground (10*23).
     * 0 indicates an empty square,
     * 1..6 are colored squares. */
    private byte[][] playground = new byte[23][10];
    
    /** The variable holding your score. */
    private int score;
    
    /** This is the x-location of the current stone. */
    private int px;
    /** This is the y-location of the current stone. */
    private int py;
    
    /** The current stone you're able to move. */
    private byte[][] thisStone = new byte[4][4];
    
    /** The next stone, as shown in the preview. */
    private byte[][] nextStone = new byte[4][4];

    /** The variables holding the seven different stones.
     * Using only the first index, they can be easily copied
     * to the arrays thisStone and nextStone. */
    private byte[][][] myStones = new byte[7][4][4];
    
    /** The in-game screen (as handled by the Thread th)
     * terminates when gameQuit is set to false. */
    private boolean gameQuit;
    
    /** This variable keeps the interval the "timer"
     * (as handled by the Thread th) uses to
     * move the stone downwards. */
    private int timerInterval; //delay in 1/1000 sec

    //int lastWinner; may be used later to indicate
    // the recent player in a highscore table.

    /** This variable determines if the game is paused or not. */
    private boolean pauseMode;
    
    /** This variable determines if the player instantly lets the
     * stone fall down. This prevents other movements to be made
     * while the fall-down-loop is active. */
    private boolean stoneIsFalling;
    
    /** This variable determines if we're in in-game-screen (0)
     * or title-screen (1). */
    private byte currentScreen=0;
    
    /** The McTris() constructor method. */
    public McTris() {
        // nothing to do
    }

    /** The main() method has no functionality at all. */
    public static void main(String[] args) {}

    /** The getAppletInfo() method can be called by the user of an
     * applet to show some general information about the applet. */
    public String getAppletInfo() {
        return "McTris v" + VERSION + " TETRIS clone, "
        + "(w) 2004 by Manuel Haim, "
        + "OPEN SOURCE, http://McSquirrel.istcool.de , "
        + "runs at 140x230 pixel, "
        + "Keys: RETURN/SPACE=start, ESC/Q=newgame, P=pause, "
        + "ARROWKEYS=left/right/rotate/letfall";
    }
    
    /** The getParameterInfo() method can be called by the user of an
     * applet to show all the supported parameters. */
    public String[][] getParameterInfo() {
        // McTris doesn't support any parameters.
        return null;
    }
    
    /** The init() method is called by the applet viewer or
     * browser when the McTris-applet is loaded. */
    public void init() {
        // Create offscreen image 140*230 and clear it:
        dbImage = createImage(10*(10+4),10*(23));
        dbGraphics = dbImage.getGraphics();

        dbGraphics.setFont(new Font("SansSerif",0,10));
        dbGraphics.setColor(java.awt.Color.black);
        dbGraphics.fillRect(0,0,10*(10+4),10*(23));
        
        repaint(); // calls the update() method
                   // (which itself calls the paint() method)

        // Initialize game variables, activate keyboard input
        // and show "welcome screen":
        initialize();
        addKeyListener(this);
        startScreen();
    }

    /** The start() method is automatically called once after
     * the init() method
     * and each time the applet gets into sight again
     * (after it has been "paused" by the stop() method)
     * or when the browser reloads the page. */
    public void start() {
        // nothing to do
    }

    /** The stop() method may be called each time the applet
     * is minimized or gets out of sight
     * or when the browser loads a different page. */
    public void stop() {
        // change to pause mode (to use less ressources):
        pauseMode = true;
    }

    /** The destroy() method is called when the applet viewer
     * or browser is closed. */
    public void destroy() {
        // nothing to do
    }

    /** The playGame() method handles gameplay.
     * The keyboard input is checked by a KeyListener. */
    private void playGame() {
        int b;
        
        // Set screen to in-game-screen:
        currentScreen = 0;
        
        // Clear the offscreen image:
        dbGraphics.setColor(java.awt.Color.black);
        dbGraphics.fillRect(0,0,10*(10+4),10*(23));
        
        // The stone's starting position:
        px = 4;
        py = 0;
        
        // Take two stones (randomly):
        java.util.Random a = new java.util.Random();
        b = a.nextInt(7); // random number 0..6
        thisStone = myStones[b]; // works wonderful!
        b = a.nextInt(7); // random number 0..6
        nextStone = myStones[b]; // works wonderful!
        
        // Set initial score and draw everything:
        score = scoreFor(6); // score for a newly inserted stone
        drawField(); // draws the playground
        drawStone(); // draws the current stone
        showNext();  // draws the next stone preview
        repaint(); // calls the update() method
                   // (which itself calls the paint() method)
        
        timerInterval = 350; // set timer interval (milliseconds)
        gameQuit = false; // game is not over yet! game has just begun!
        pauseMode = false; // we're not in pause mode!
        stoneIsFalling = false; // stone isn't yet to fall down!
        
        th = new Thread(this);
        th.start(); // calls the run() method
        // The "Game" now runs as Thread th in background.
        
    }
    
    /** The startScreen() method activates the
     * welcome screen. */
    private void startScreen() {
        // Set screen to welcome-screen:
        currentScreen = 1;
        repaint(); // calls the update() method
                   // (which itself calls the paint() method)
    }

    /** The run() method is used for handling an object of this class
     * as a thread. This thread does the "timer" work and moves the
     * stone downwards. */
    public void run() {
        do {
            if((!pauseMode)&&(!stoneIsFalling)) {
                stoneMove(0,1); // move stone one step down
            }
            try {
                Thread.sleep(timerInterval); // wait .. ms
            } catch (InterruptedException e) {
                // nothing
            }
        } while(!gameQuit);
        
        // The game has been quit.
        addHighscore();
        saveHighscore();
        
        // clear playground:
        for(int y=0;y<23;y++) {
            for(int x=0;x<10;x++) {
                playground[y][x] = 0;
            }
        }

        // go back to "welcome screen":
        startScreen();
        // And now this thread terminates.
    }

    /** This method paints the image onto the screen. It is called by the
     * system or by the update() method (which itself is called by the
     * repaint() method). */
    public void paint(Graphics g) {
        
        /** All drawing is done to an offscreen image. Finally, when this
         * image is complete, it is drawn onto the screen. This way we
         * reduce flickering. */
        switch(currentScreen) {
        case 0:
            // Draw the "in-game screen":
            /** The stones are already painted by the drawStone(),
             * drawField() and showNext() methods. We only have to
             * add the highscore here. */
            dbGraphics.setColor(java.awt.Color.black);
            dbGraphics.fillRect(10*(10),10*(4),10*(4),10*(1));
            dbGraphics.setColor(java.awt.Color.white);
            dbGraphics.drawString((new Integer(score)).toString(),
            10*(10),10*(5));
            
            /** Pause mode shows "Paused" on the screen: */
            if(pauseMode) {
                dbGraphics.setColor(java.awt.Color.black);
                dbGraphics.fillRect(
                10*(3),
                10*(8),
                10*(4),
                10*(1)+2
                );
                dbGraphics.setColor(java.awt.Color.white);
                dbGraphics.drawString("Paused.",
                10*(3),10*(9));
            }

            // Finally draw the offscreen image onto the screen:
            g.drawImage(dbImage,0,0,this);
            break;
        case 1:
            // Draw the "welcome screen":
            dbGraphics.setColor(java.awt.Color.black);
            dbGraphics.fillRect(0,0,10*(10+4),10*(23));
            dbGraphics.setColor(java.awt.Color.green);
            dbGraphics.setFont(new Font("SansSerif",1,20));
            dbGraphics.drawString("McTris v" + VERSION,10*(1),10*(3));
            dbGraphics.setColor(java.awt.Color.yellow);
            dbGraphics.setFont(new Font("SansSerif",0,10));
            dbGraphics.drawString("\"The somewhat",10*(3),10*(5));
            dbGraphics.drawString("different Tetris.\"",10*(3),10*(6));
            dbGraphics.drawString("(w) by Manuel Haim",10*(2),10*(8));
            dbGraphics.drawString("in 2004",10*(5),10*(9));
            dbGraphics.drawString("Last score: "
            + (new Integer(score)).toString(),
            10*(2),10*(15));
            
            dbGraphics.setColor(java.awt.Color.blue);
            dbGraphics.drawString("-Open Source Release-",
            10*(1)+5,10*(18));
            dbGraphics.drawString("http://McSquirrel.istcool.de",10*(0)+5,
            10*(19));

            dbGraphics.setColor(java.awt.Color.white);
            dbGraphics.drawString("Press Return to play game.",
            10*(1),10*(22));
            
            // Finally draw the offscreen image onto the screen:
            g.drawImage(dbImage,0,0,this);
            break;
        }
    }

    /** The update() method is called each time we (or the system) call the
     * repaint() method. Normally, the update() method clears the screen
     * and then calls the paint() method. This way, the applet would seem to
     * flicker, as we have a blank image before each paint().
     * That's why we overwrite the update() method. The way we go is called
     * "double-buffering". All clearing and drawing is done to an offscreen
     * image which is finally drawn by the paint() method when it's ready
     * (WITHOUT clearing the previously shown image). */
    public void update(Graphics g) {
        // Just one call to the paint() method (no screen-clearing):
        paint(g);
    }

    /** The KeyListener method keyPressed() is automatically called when
     * the user pushs a key down. */
    public void keyPressed(KeyEvent event) {
        switch(currentScreen) {
        case 0: // in-game screen
            if(!pauseMode) {
                switch(event.getKeyCode()) {
                case KeyEvent.VK_LEFT:
                    if(stoneIsFalling) {break;}
                    stoneMove(-1,0);
                    break;
                case KeyEvent.VK_RIGHT:
                    if(stoneIsFalling) {break;}
                    stoneMove(1,0);
                    break;
                case KeyEvent.VK_UP:
                    if(stoneIsFalling) {break;}
                    stoneRotate();
                    break;
                case KeyEvent.VK_DOWN:
                    if(stoneIsFalling) {break;}
                    stoneIsFalling = true;
                    do {
                        /** This may cause errors, as when
                         * we're in this loop, the event VK_DOWN
                         * could occur another time (so the loop
                         * maybe started twice or more times!).
                         * Maybe a new method "stoneFall" would
                         * help which checks if it's already
                         * running. */
                        // add score for falling down one line:
                        score+=scoreFor(5);
                        // move stone one line down:
                        stoneMove(0,1);
                    } while((py!=0)&&(!gameQuit));
                    stoneIsFalling = false;
                    break;
                case KeyEvent.VK_P:
                    pauseMode = true;
                    break;
                case KeyEvent.VK_ESCAPE:
                case KeyEvent.VK_Q:
                    gameQuit = true;
                    break;
                default:
                }
            }
            else {
                pauseMode = false;
            }
            break;
        case 1: // "welcome screen"
            switch(event.getKeyCode()) {
            case KeyEvent.VK_ENTER:
            case KeyEvent.VK_SPACE:
                playGame(); // sets currentScreen=0 and starts game
            }
        }
        
        repaint();
    }

    /** The KeyListener method keyReleased() is automatically called when
     * the user lets a key up. */
    public void keyReleased(KeyEvent event) {
    }

    /** The KeyListener method keyTyped() is automatically called when
     * the user enters a character. */
    public void keyTyped(KeyEvent event) {
    }

    /** NB: An applet can neither load nor save the highscores
     * to the user's local disk, so these methods are empty
     * here. */
     
    /** The loadHighscore() method is intended to load a highscore
     * table from the local disk or a server at game startup. */
    private void loadHighscore() {}
    
    /** The addHighscore() method is intended to add the recent
     * score to the highscore table. It also has to ask the player
     * for his name. It could be used to transmit the player's score
     * to a server. */
    private void addHighscore() {}
    
    /** The saveHighscore() method is intended to save the highscore
     * table to the local disk each time the table changes. It would
     * make no sense to transmit the complete table to a server here,
     * but when using a server you should call the loadHighscore()
     * method from within this method to show an up-to-date table to
     * the player. */
    private void saveHighscore() {}

    /** The scoreFor() method returns the score for an action.
     * @param s Number of the action:
     * 1/2/3/4 = completing 1/2/3/4 lines,
     * 5 = fall down one line,
     * 6 = insert new stone.
     * @return Score for that action. */
    private int scoreFor(int s) {
        switch(s) {
        case 1: // score for completing one line:
            return 100;
        case 2: // score for completing two lines:
            return 200;
        case 3: // score for completing three lines:
            return 400;
        case 4: // score for completing four lines:
            return 800;
        case 5: // score for falling down one line:
            return 5;
        case 6: // score for a newly inserted stone:
            return 10;
        default:
            return 0;
        }
    }

    /** The stoneColor() method returns the color of a stone.
     * @param c Stone number.
     * @return Color of that stone. */
    private java.awt.Color stoneColor(byte c) {
        switch(c) {
        case 1:
            return java.awt.Color.lightGray;
        case 2:
            return java.awt.Color.blue;
        case 3:
            return java.awt.Color.red;
        case 4:
            // purple:
            return new java.awt.Color(0.5f,0.0f,0.5f);
        case 5:
            return java.awt.Color.green;
        case 6:
            return java.awt.Color.yellow;
        case 7:
            return java.awt.Color.cyan;
        default:
            return java.awt.Color.black;
        }
    }

    /** The drawField() method draws the currently existing
     * stones into the offscreen image (square-by-square). */
    private void drawField() {
        // paint gray background:
        dbGraphics.setColor(java.awt.Color.gray);
        dbGraphics.fillRect(0,0,10*(10),10*(23));
        // paint field:
        for(int y=0;y<23;y++) {
            for(int x=0;x<10;x++) {
                if(playground[y][x]!=0) {
                    dbGraphics.setColor(stoneColor(playground[y][x]));
                    dbGraphics.fill3DRect(x*10,y*10,10,10,true);
                }
            }
        }
    }
    
    /** The drawStone() method draws the currently falling
     * stone onto the offscreen image (square-by-square). */
    private void drawStone() {
        // paint stone:
        for(int y=0;y<4;y++) {
            for(int x=0;x<4;x++) {
                if(thisStone[y][x]!=0) {
                    dbGraphics.setColor(stoneColor(thisStone[y][x]));
                    dbGraphics.fill3DRect((x+px)*10,(y+py)*10,10,10,true);
                }
            }
        }
        
    }
    
    /** The showNext() method draws the preview of the next
     * stone into the offscreen image (square-by-square). */
    private void showNext() {
        // paint black background:
        dbGraphics.setColor(java.awt.Color.black);
        dbGraphics.fillRect(10*(10),0,10*(4),10*(4));
        // paint stone:
        for(int y=0;y<4;y++) {
            for(int x=0;x<4;x++) {
                if(nextStone[y][x]!=0) {
                    dbGraphics.setColor(stoneColor(nextStone[y][x]));
                    dbGraphics.fill3DRect((x+10)*10,y*10,10,10,true);
                }
            }
        }
    }
    
    /** The method stoneValid() checks square-by-square, whether a
     * stone fits into the playground matrix at a given position or not.
     * @param stone The stone to be checked.
     * @param x1 The x-coordinate of the stone's upper left corner.
     * @param y1 The y-coordinate of the stone's upper left corner.
     * @return true if stone fits, otherwise false. */
    private boolean stoneValid(byte[][] stone, int x1, int y1) {
        boolean b1 = true;
        
        for(int y=y1;y<y1+4;y++) {
            for(int x=x1;x<x1+4;x++) {
                // if the current square is inside the playground:
                if((x>=0)&&(x<10)&&(y>=0)&&(y<23)) {
                    // if one part of the stone would overlap another
                    // existing stone:
                    if(thisStone[y-y1][x-x1]!=0) {
                        if(playground[y][x]!=0) {
                            b1 = false;
                        }
                    }
                }
                else
                // if the current square is not inside the playground,
                // it must be empty to fit:
                if(thisStone[y-y1][x-x1]!=0) {
                    b1 = false;
                }
            }
        }
        return b1;
    }

    /** The stoneMove() method moves the stone within the
     * playground matrix (by changing its px and py coordinate).
     * This is only done if the stone fits into the new position.
     * @param x1 1 moves the stone one step right (if possible),
     * -1 moves the stone one step left (if possible).
     * @param y1 1 moves the stone one step down (if possible),
     * otherwise the stone gets fixed within the playground matrix. */
    private void stoneMove(int x1, int y1) {
        boolean b1;
        boolean b2;
        int z2;
        
        if(x1==1) {
            if(stoneValid(thisStone,px+1,py)) {
                // move stone one step right:
                px++;
                drawField();
                drawStone();
                repaint();
            }
        }
        if(x1==-1) {
            if(stoneValid(thisStone,px-1,py)) {
                // move stone one step left:
                px--;
                drawField();
                drawStone();
                repaint();
            }
        }
        if(y1==1) {
            if(stoneValid(thisStone,px,py+1)) {
                // move stone one step down:
                py++;
                drawField();
                drawStone();
                repaint();
            }
            else {
                // put stone (square-by-square) into playground matrix:
                for(int y=py;y<py+4;y++) {
                    for(int x=px;x<px+4;x++) {
                        if(thisStone[y-py][x-px]!=0) {
                            playground[y][x] = thisStone[y-py][x-px];
                        }
                    }
                }
                // clear completed lines:
                z2 = 0;
                for(int y=0;y<23;y++) {
                    b2 = true;
                    for(int x=0;x<10;x++) {
                        if(playground[y][x]==0) {
                            b2 = false;
                        }
                    }
                    if(b2) {
                        for(int x=0;x<10;x++) {
                            playground[y][x] = 0;
                        }
                        z2++;
                        drawField();
                        drawStone();
                        repaint();
                        for(int z1=y;z1>=1;z1--) {
                            for(int x=0;x<10;x++) {
                                playground[z1][x] = playground[z1-1][x];
                            }
                        }
                        for(int x=0;x<10;x++) {
                            playground[0][x] = 0;
                        }
                        drawField();
                        drawStone();
                        repaint();
                    }
                }
                // add points to score:
                score+=scoreFor(z2);
                // insert a new stone:
                thisStone = nextStone;
                if(!stoneValid(thisStone,4,0)) {
                    // Game Over:
                    gameQuit = true;
                }
                else { // Game not over:
                    score+=scoreFor(6); // score for a new stone
                    java.util.Random a = new java.util.Random();
                    int b = a.nextInt(7); // random number 0..6
                    nextStone = myStones[b];
                    showNext();
                    py = 0;
                    px = 4;
                    drawField();
                    drawStone();
                    repaint();
                }
            }
        }
    }
    
    /** The stoneRotate() method rotates the stone clockwise.
     * If the stone would hurt the right border after rotating,
     * the stone is moved some steps left (if possible). */
    private void stoneRotate() {
        byte[][] tempStone = new byte[4][4];
        boolean b1;
        boolean towardsCenter;
        int v_px = px;
        int v_py = py;
        
        // this makes tempStone become a rotated copy of the stone:
        for(int y=0;y<4;y++) {
            for(int x=0;x<4;x++) {
                tempStone[y][x] = thisStone[3-x][y];
            }
        }
        // align-to-top:
        for(int z1=0;z1<3;z1++) {
            b1 = true;
            for(int x=0;x<4;x++) {
                if(tempStone[0][x]!=0) {
                    b1 = false;
                }
            }
            if(b1==true) {
                for(int y=1;y<4;y++) {
                    for(int x=0;x<4;x++) {
                        tempStone[y-1][x] = tempStone[y][x];
                    }
                }
                for(int x=0;x<4;x++) {
                    tempStone[3][x] = 0;
                }
            }
        }
        // align-to-left:
        for(int z1=0;z1<3;z1++) {
            b1 = true;
            for(int y=0;y<4;y++) {
                if(tempStone[y][0]!=0) {
                    b1 = false;
                }
            }
            if(b1==true) {
                for(int y=0;y<4;y++) {
                    for(int x=1;x<4;x++) {
                        tempStone[y][x-1] = tempStone[y][x];
                    }
                }
                for(int y=0;y<4;y++) {
                    tempStone[y][3] = 0;
                }
            }
        }
        // Now testing, if tempStone is suitable:
        
        // "label" towardsCenter: (not possible in Java)
        towardsCenter = false;
        do {
            towardsCenter = false;
            b1 = true;
            
            // check square-by-square if the rotated stone fits:
            for(int y=v_py;y<v_py+4;y++) {
                for(int x=v_px;x<v_px+4;x++) {
                    // if the current square is inside the playground:
                    if((x>=0)&&(x<10)&&(y>=0)&&(y<23)) {
                        // if a part of the stone would overlap another
                        // existing stone:
                        if(tempStone[y-v_py][x-v_px]!=0) {
                            if(playground[y][x]!=0) {
                                b1 = false;
                            }
                        }
                    }
                    else {
                        // if, after rotating the stone, a part of it would be
                        // out of the playground:
                        // try, if the stone is at the right border, to move
                        // it some units left, till nothing is out of the
                        // playground.
                        if((y<0)||(y>=23)) {
                            b1 = false;
                        }
                        else
                        if(tempStone[y-v_py][x-v_px]!=0) {
                            if(x>=10) {
                                v_px--;
                            }
                            // if(x<0){v_px++;} doesn't happen, as stone
                            // is align-left.
                            
                            // "goto" towardsCenter; (not possible in Java)
                            towardsCenter = true;
                        }
                    }
                    if(towardsCenter) {break;}
                }
                if(towardsCenter) {break;}
            }
        } while(towardsCenter == true);
        // now insert
        if(b1==true) {
            thisStone = tempStone;
            px = v_px;
            py = v_py;
            drawField();
            drawStone();
            repaint();
        }
    }
    
    /** The initialize() method is to be called at program startup.
     * It initializes several important variables. */
    private void initialize() {
        loadHighscore();
        
        for(int y1=0;y1<23;y1++){
            for(int x1=0;x1<10;x1++) {
                playground[y1][x1] = 0;
            }
        }
        
        for(int y1=0;y1<4;y1++) {
            for(int x1=0;x1<4;x1++) {
                thisStone[y1][x1] = 0;
                nextStone[y1][x1] = 2;
            }
        }
        
        // Here the "main thing", the seven stones:
        myStones[0][0][0] = 0;
        myStones[0][0][1] = 1;
        myStones[0][0][2] = 0;
        myStones[0][0][3] = 0;
        myStones[0][1][0] = 1;
        myStones[0][1][1] = 1;
        myStones[0][1][2] = 1;
        myStones[0][1][3] = 0;
        myStones[0][2][0] = 0;
        myStones[0][2][1] = 0;
        myStones[0][2][2] = 0;
        myStones[0][2][3] = 0;
        myStones[0][3][0] = 0;
        myStones[0][3][1] = 0;
        myStones[0][3][2] = 0;
        myStones[0][3][3] = 0;
        /**
        (((0,1,0,0),
          (1,1,1,0),
          (0,0,0,0),
          (0,0,0,0)),**/
        
        myStones[1][0][0] = 2;
        myStones[1][0][1] = 2;
        myStones[1][0][2] = 2;
        myStones[1][0][3] = 0;
        myStones[1][1][0] = 2;
        myStones[1][1][1] = 0;
        myStones[1][1][2] = 0;
        myStones[1][1][3] = 0;
        myStones[1][2][0] = 0;
        myStones[1][2][1] = 0;
        myStones[1][2][2] = 0;
        myStones[1][2][3] = 0;
        myStones[1][3][0] = 0;
        myStones[1][3][1] = 0;
        myStones[1][3][2] = 0;
        myStones[1][3][3] = 0;
        /**
        ((2,2,2,0),
         (2,0,0,0),
         (0,0,0,0),
         (0,0,0,0)),**/
        
        myStones[2][0][0] = 3;
        myStones[2][0][1] = 3;
        myStones[2][0][2] = 3;
        myStones[2][0][3] = 0;
        myStones[2][1][0] = 0;
        myStones[2][1][1] = 0;
        myStones[2][1][2] = 3;
        myStones[2][1][3] = 0;
        myStones[2][2][0] = 0;
        myStones[2][2][1] = 0;
        myStones[2][2][2] = 0;
        myStones[2][2][3] = 0;
        myStones[2][3][0] = 0;
        myStones[2][3][1] = 0;
        myStones[2][3][2] = 0;
        myStones[2][3][3] = 0;
        /**
        ((3,3,3,0),
         (0,0,3,0),
         (0,0,0,0),
         (0,0,0,0)),**/
        
        myStones[3][0][0] = 4;
        myStones[3][0][1] = 4;
        myStones[3][0][2] = 0;
        myStones[3][0][3] = 0;
        myStones[3][1][0] = 0;
        myStones[3][1][1] = 4;
        myStones[3][1][2] = 4;
        myStones[3][1][3] = 0;
        myStones[3][2][0] = 0;
        myStones[3][2][1] = 0;
        myStones[3][2][2] = 0;
        myStones[3][2][3] = 0;
        myStones[3][3][0] = 0;
        myStones[3][3][1] = 0;
        myStones[3][3][2] = 0;
        myStones[3][3][3] = 0;
        /**
        ((4,4,0,0),
         (0,4,4,0),
         (0,0,0,0),
         (0,0,0,0)),**/
        
        myStones[4][0][0] = 0;
        myStones[4][0][1] = 5;
        myStones[4][0][2] = 5;
        myStones[4][0][3] = 0;
        myStones[4][1][0] = 5;
        myStones[4][1][1] = 5;
        myStones[4][1][2] = 0;
        myStones[4][1][3] = 0;
        myStones[4][2][0] = 0;
        myStones[4][2][1] = 0;
        myStones[4][2][2] = 0;
        myStones[4][2][3] = 0;
        myStones[4][3][0] = 0;
        myStones[4][3][1] = 0;
        myStones[4][3][2] = 0;
        myStones[4][3][3] = 0;
        /**
        ((0,5,5,0),
         (5,5,0,0),
         (0,0,0,0),
         (0,0,0,0)),**/
        
        myStones[5][0][0] = 6;
        myStones[5][0][1] = 6;
        myStones[5][0][2] = 6;
        myStones[5][0][3] = 6;
        myStones[5][1][0] = 0;
        myStones[5][1][1] = 0;
        myStones[5][1][2] = 0;
        myStones[5][1][3] = 0;
        myStones[5][2][0] = 0;
        myStones[5][2][1] = 0;
        myStones[5][2][2] = 0;
        myStones[5][2][3] = 0;
        myStones[5][3][0] = 0;
        myStones[5][3][1] = 0;
        myStones[5][3][2] = 0;
        myStones[5][3][3] = 0;
        /**
        ((6,6,6,6),
         (0,0,0,0),
         (0,0,0,0),
         (0,0,0,0)),**/
        
        myStones[6][0][0] = 7;
        myStones[6][0][1] = 7;
        myStones[6][0][2] = 0;
        myStones[6][0][3] = 0;
        myStones[6][1][0] = 7;
        myStones[6][1][1] = 7;
        myStones[6][1][2] = 0;
        myStones[6][1][3] = 0;
        myStones[6][2][0] = 0;
        myStones[6][2][1] = 0;
        myStones[6][2][2] = 0;
        myStones[6][2][3] = 0;
        myStones[6][3][0] = 0;
        myStones[6][3][1] = 0;
        myStones[6][3][2] = 0;
        myStones[6][3][3] = 0;
        /**
        ((7,7,0,0),
         (7,7,0,0),
         (0,0,0,0),
         (0,0,0,0)));**/
        
        // lastWinner = 0; may be used later to indicate
        // the recent player in a highscore table.
        
    }

}
